<?php
/**
 * Ethereum MEV Analyzer - Maximal Extractable Value Intelligence
 * Purpose: Analyze MEV activity and protect users from MEV attacks
 * Analysis: MEV rate calculation, user risk scoring, sandwich detection
 * Output: Protection recommendations with threat assessment
 */

class MEVAnalyzer {
    private $db;
    private $flashbots_api_key;
    private $etherscan_api_key;
    
    public function __construct() {
        $this->db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                           "nvdaxcom_galliumhash_metabuilder_user", 
                           "32TbgulnE7YOd2G0e5");
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // API keys
        $this->flashbots_api_key = "f3ZVHsvpZAMU-jDKbX_mk"; // Using Alchemy as Flashbots endpoint
        $this->etherscan_api_key = "ZR2CAV375V5R3GCHWYVNZNBZZGTNZ5KH78";
    }
    
    /**
     * Analyze MEV activity level across the network
     */
    private function analyzeMEVActivity() {
        $recent_mev_data = $this->getRecentMEVData(1000); // Last 1000 blocks
        
        if (empty($recent_mev_data)) {
            // Fallback to proxy analysis using transaction patterns
            return $this->analyzeTransactionPatterns();
        }
        
        $mev_extracted = array_column($recent_mev_data, 'mev_extracted_eth');
        $mev_per_block = array_sum($mev_extracted) / count($recent_mev_data);
        
        // Calculate MEV intensity ratio
        $avg_mev_per_block = $mev_per_block;
        $recent_mev_10 = array_sum(array_slice($mev_extracted, 0, 10)) / 10;
        $mev_intensity_ratio = $avg_mev_per_block > 0 ? $recent_mev_10 / $avg_mev_per_block : 1;
        
        // Classify activity level
        if ($mev_intensity_ratio > 3) {
            $activity_level = "CRITICAL";
        } elseif ($mev_intensity_ratio > 2) {
            $activity_level = "ELEVATED";
        } elseif ($mev_intensity_ratio < 0.5) {
            $activity_level = "LOW";
        } else {
            $activity_level = "MODERATE";
        }
        
        return [
            'mev_intensity_ratio' => $mev_intensity_ratio,
            'activity_level' => $activity_level,
            'avg_mev_per_block' => $avg_mev_per_block,
            'recent_mev_trend' => $this->calculateMEVTrend($mev_extracted),
            'total_extracted_24h' => $this->calculateTotalExtracted24h(),
            'active_bots_estimate' => $this->estimateActiveBots()
        ];
    }
    
    /**
     * Calculate user transaction risk scoring by transaction type
     */
    private function calculateTransactionRisk($transaction_type, $trade_size = null, $slippage_tolerance = 1.0) {
        $risk_data = [];
        
        switch ($transaction_type) {
            case 'eth_transfer':
                $risk_data = [
                    'risk_score' => 0.1,
                    'risk_level' => 'MINIMAL',
                    'expected_loss' => 0,
                    'mev_types' => ['none'],
                    'protection_needed' => false
                ];
                break;
                
            case 'erc20_transfer':
                $risk_data = [
                    'risk_score' => 0.2,
                    'risk_level' => 'LOW',
                    'expected_loss' => 0.5,
                    'mev_types' => ['minimal_front_running'],
                    'protection_needed' => false
                ];
                break;
                
            case 'dex_swap':
                $risk_data = $this->analyzeDEXSwapRisk($trade_size, $slippage_tolerance);
                break;
                
            case 'nft_purchase':
                $risk_data = $this->analyzeNFTPurchaseRisk($trade_size);
                break;
                
            case 'liquidation':
                $risk_data = [
                    'risk_score' => 0.3,
                    'risk_level' => 'MODERATE',
                    'expected_loss' => 0,
                    'mev_types' => ['competition_for_liquidation'],
                    'protection_needed' => false,
                    'note' => 'MEV bots compete but user gets liquidated anyway'
                ];
                break;
                
            default:
                $risk_data = [
                    'risk_score' => 0.5,
                    'risk_level' => 'MODERATE',
                    'expected_loss' => 10.0,
                    'mev_types' => ['unknown'],
                    'protection_needed' => true
                ];
        }
        
        return $risk_data;
    }
    
    /**
     * Analyze DEX swap risk with specific factors
     */
    private function analyzeDEXSwapRisk($trade_size, $slippage_tolerance) {
        // Get current gas prices for bot profitability calculation
        $current_gas = $this->getCurrentGasPrice();
        $eth_price = $this->getEthPrice();
        
        // Default values if not specified
        $trade_size = $trade_size ?: 1000; // $1000 default
        $slippage_tolerance = $slippage_tolerance ?: 1.0; // 1% default
        
        // Simulate pool liquidity and price impact
        $pool_liquidity = $this->simulatePoolLiquidity();
        $price_impact = ($trade_size / $pool_liquidity) * 100; // Simplified price impact
        
        // Calculate sandwich attack probability
        $sandwich_probability = $this->calculateSandwichProbability(
            $trade_size, 
            $slippage_tolerance, 
            $current_gas, 
            $pool_liquidity
        );
        
        // Calculate expected MEV loss
        $expected_loss = $this->calculateExpectedSandwichLoss($trade_size, $price_impact);
        
        // Determine risk level based on multiple factors
        $risk_score = 0.3; // Base risk
        
        if ($price_impact > 1) {
            $risk_score += 0.3 + ($price_impact * 0.1);
        } elseif ($price_impact > 0.5) {
            $risk_score += 0.2 + ($price_impact * 0.2);
        }
        
        if ($slippage_tolerance > 1) {
            $risk_score += ($slippage_tolerance - 1) * 0.2;
        }
        
        if ($sandwich_probability > 0.7) {
            $risk_score += 0.2;
        }
        
        $risk_score = min(1.0, $risk_score);
        
        // Determine protection needed
        $protection_needed = $risk_score > 0.5 || $sandwich_probability > 0.6;
        
        $risk_level = $risk_score > 0.7 ? 'HIGH' : 
                     ($risk_score > 0.4 ? 'MODERATE' : 'LOW');
        
        return [
            'risk_score' => round($risk_score, 2),
            'risk_level' => $risk_level,
            'expected_loss' => round($expected_loss, 2),
            'sandwich_probability' => round($sandwich_probability, 2),
            'price_impact' => round($price_impact, 2),
            'pool_liquidity' => $pool_liquidity,
            'mev_types' => ['sandwich_attack', 'back_running'],
            'protection_needed' => $protection_needed,
            'recommendations' => $this->generateSwapRecommendations($risk_score, $sandwich_probability)
        ];
    }
    
    /**
     * Analyze NFT purchase risk
     */
    private function analyzeNFTPurchaseRisk($purchase_value) {
        $purchase_value = $purchase_value ?: 100; // $100 default
        
        // Check for popular mint or high-value NFT
        $rarity_score = $this->estimateNFTRarity($purchase_value);
        
        $sniping_probability = 0;
        if ($rarity_score > 90) {
            $sniping_probability = 0.65; // High sniping risk for rare NFTs
        } elseif ($rarity_score > 70) {
            $sniping_probability = 0.4;
        } else {
            $sniping_probability = 0.2;
        }
        
        $expected_loss = $purchase_value * $sniping_probability * 0.1; // 10% loss estimate
        
        $risk_score = 0.4 + ($sniping_probability * 0.4);
        
        return [
            'risk_score' => round($risk_score, 2),
            'risk_level' => $risk_score > 0.7 ? 'HIGH' : 'MODERATE',
            'expected_loss' => round($expected_loss, 2),
            'sniping_probability' => round($sniping_probability, 2),
            'rarity_score' => $rarity_score,
            'mev_types' => ['nft_sniping', 'front_running'],
            'protection_needed' => $sniping_probability > 0.4,
            'recommendations' => [
                'Use private mint services',
                'Reduce slippage tolerance',
                'Consider limit orders'
            ]
        ];
    }
    
    /**
     * Calculate sandwich attack probability
     */
    private function calculateSandwichProbability($trade_size, $slippage_tolerance, $gas_price, $pool_liquidity) {
        $probability = 0;
        
        // Factor 1: Trade size vs pool liquidity
        $trade_ratio = $trade_size / $pool_liquidity;
        if ($trade_ratio > 0.01) { // 1% of pool
            $probability += 0.3;
        }
        
        // Factor 2: Slippage tolerance
        if ($slippage_tolerance > 1) {
            $probability += 0.2;
        }
        
        // Factor 3: Gas price (bots need profit after gas)
        $current_gas_gwei = $gas_price / 1e9;
        if ($current_gas_gwei < 30) { // Low gas = profitable for bots
            $probability += 0.2;
        }
        
        // Factor 4: Pool depth
        if ($pool_liquidity < 100000) { // $100k pool
            $probability += 0.3;
        }
        
        // Factor 5: Current MEV activity
        $mev_intensity = $this->getCurrentMEVIntensity();
        if ($mev_intensity > 2) {
            $probability += 0.2;
        }
        
        return min(1.0, $probability);
    }
    
    /**
     * Calculate expected sandwich attack loss
     */
    private function calculateExpectedSandwichLoss($trade_size, $price_impact) {
        // Simplified sandwich attack simulation
        $front_run_impact = $price_impact * 1.5; // Bots buy first
        $back_run_impact = $price_impact * 0.8;  // Bots sell after
        
        $total_impact = ($front_run_impact + $back_run_impact) / 2;
        $expected_loss = $trade_size * ($total_impact / 100);
        
        return max(0, $expected_loss);
    }
    
    /**
     * Analyze MEV bot landscape
     */
    private function analyzeMEVBots() {
        // In real implementation, this would analyze MEV bot data from Flashbots
        $mock_bot_data = $this->getMockBotData();
        
        $bot_intelligence = [];
        $total_profit = 0;
        $total_victims = 0;
        
        foreach ($mock_bot_data as $bot) {
            // Classify bot strategy
            $strategy = $this->classifyBotStrategy($bot);
            
            $success_rate = $bot['successful_mev'] / $bot['total_attempts'];
            $avg_value_extracted = $bot['total_profit'] / $bot['victims_count'];
            
            $bot_intelligence[] = [
                'address' => $bot['address'],
                'strategy' => $strategy,
                'profit_24h' => $bot['total_profit'],
                'success_rate' => round($success_rate, 2),
                'victims_24h' => $bot['victims_count'],
                'avg_extraction' => round($avg_value_extracted, 2),
                'threat_level' => $this->classifyBotThreat($bot)
            ];
            
            $total_profit += $bot['total_profit'];
            $total_victims += $bot['victims_count'];
        }
        
        // Sort by threat level
        usort($bot_intelligence, function($a, $b) {
            $threat_order = ['CRITICAL' => 4, 'HIGH' => 3, 'MODERATE' => 2, 'LOW' => 1];
            return $threat_order[$b['threat_level']] <=> $threat_order[$a['threat_level']];
        });
        
        return [
            'active_bots_24h' => count($bot_intelligence),
            'top_bots' => array_slice($bot_intelligence, 0, 5),
            'total_profit_24h' => round($total_profit, 2),
            'total_victims_24h' => $total_victims,
            'strategy_breakdown' => $this->getStrategyBreakdown($bot_intelligence)
        ];
    }
    
    /**
     * Assess protocol MEV vulnerability
     */
    private function assessProtocolVulnerability() {
        $protocols = $this->getProtocolData();
        $vulnerability_assessment = [];
        
        foreach ($protocols as $protocol) {
            $mev_data = $this->getProtocolMEVData($protocol['name']);
            
            $total_attacks = $mev_data['attacks_30d'] ?? 0;
            $total_value = $mev_data['value_extracted_30d'] ?? 0;
            $unique_victims = $mev_data['users_affected_30d'] ?? 0;
            
            // Risk classification
            if ($total_attacks > 1000) {
                $risk = "HIGH";
            } elseif ($total_attacks > 100) {
                $risk = "MEDIUM";
            } else {
                $risk = "LOW";
            }
            
            $vulnerability_assessment[] = [
                'protocol' => $protocol['name'],
                'mev_risk' => $risk,
                'attacks_24h' => $mev_data['attacks_24h'] ?? 0,
                'attacks_30d' => $total_attacks,
                'value_extracted_24h' => $mev_data['value_extracted_24h'] ?? 0,
                'value_extracted_30d' => $total_value,
                'users_affected_24h' => $mev_data['users_affected_24h'] ?? 0,
                'users_affected_30d' => $unique_victims,
                'most_common_attack' => $mev_data['common_attack'] ?? 'sandwich',
                'protection_available' => $this->checkMEVProtection($protocol['name'])
            ];
        }
        
        return $vulnerability_assessment;
    }
    
    /**
     * Generate MEV protection recommendations
     */
    private function generateProtectionRecommendations($risk_score, $expected_loss, $transaction_type) {
        $recommendations = [];
        
        if ($risk_score < 0.3) {
            return [
                'protection_needed' => false,
                'reason' => 'Low MEV risk for this transaction type',
                'recommendation' => 'No special protection required'
            ];
        }
        
        if ($risk_score < 0.6) {
            $recommendations[] = "Reduce slippage tolerance to 0.5%";
            $recommendations[] = "Split trade into smaller chunks";
            $recommendations[] = "Consider timing (low gas periods = more MEV)";
            
            return [
                'protection_needed' => true,
                'severity' => 'MODERATE',
                'reason' => "Estimated MEV loss: $" . round($expected_loss, 2),
                'recommendations' => $recommendations
            ];
        }
        
        // High risk
        $recommendations = [
            'Use MEV protection RPC (Flashbots Protect)',
            'Split into multiple smaller trades',
            'Use limit orders instead of market orders',
            'Consider CoW Protocol (MEV-resistant DEX)',
            'Reduce slippage tolerance to 0.3%'
        ];
        
        return [
            'protection_needed' => true,
            'severity' => 'HIGH',
            'reason' => "High sandwich attack risk, estimated loss: $" . round($expected_loss, 2),
            'recommendations' => $recommendations,
            'alternative_services' => $this->getMEVProtectionServices()
        ];
    }
    
    /**
     * Get MEV protection services
     */
    private function getMEVProtectionServices() {
        return [
            [
                'name' => 'Flashbots Protect',
                'type' => 'RPC',
                'cost' => 'Free',
                'effectiveness' => 'High',
                'url' => 'https://rpc.flashbots.net'
            ],
            [
                'name' => 'CoW Protocol',
                'type' => 'DEX',
                'cost' => 'Standard swap fees',
                'effectiveness' => 'Very High',
                'url' => 'https://cow.fi'
            ],
            [
                'name' => '1inch Fusion',
                'type' => 'DEX Aggregator',
                'cost' => 'Standard swap fees',
                'effectiveness' => 'High',
                'url' => 'https://1inch.io'
            ]
        ];
    }
    
    /**
     * Calculate MEV trend
     */
    private function calculateMEVTrend($mev_data) {
        if (count($mev_data) < 20) return "STABLE";
        
        $recent = array_slice($mev_data, 0, 10);
        $older = array_slice($mev_data, 10, 10);
        
        $recent_avg = array_sum($recent) / count($recent);
        $older_avg = array_sum($older) / count($older);
        
        $change_percent = (($recent_avg - $older_avg) / $older_avg) * 100;
        
        if ($change_percent > 20) return "INCREASING";
        if ($change_percent < -20) return "DECREASING";
        return "STABLE";
    }
    
    /**
     * Helper functions (mock implementations)
     */
    private function getRecentMEVData($limit) {
        // In real implementation, this would query MEV data from database
        return [];
    }
    
    private function analyzeTransactionPatterns() {
        // Proxy analysis using transaction data patterns
        return [
            'mev_intensity_ratio' => 1.2,
            'activity_level' => 'MODERATE',
            'avg_mev_per_block' => 0.034,
            'recent_mev_trend' => 'STABLE'
        ];
    }
    
    private function getCurrentGasPrice() {
        try {
            $stmt = $this->db->query("SELECT base_fee FROM ethereum_intelligence ORDER BY block_number DESC LIMIT 1");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result['base_fee'] * 1e9; // Convert from gwei to wei
        } catch (Exception $e) {
            return 35e9; // Default 35 gwei
        }
    }
    
    private function getEthPrice() {
        try {
            $stmt = $this->db->query("SELECT eth_price_usd FROM user_activity_summary LIMIT 1");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result['eth_price_usd'] ?: 2300.0;
        } catch (Exception $e) {
            return 2300.0;
        }
    }
    
    private function simulatePoolLiquidity() {
        return rand(100000, 1000000); // Random pool size between $100k-$1M
    }
    
    private function getCurrentMEVIntensity() {
        return 1.5; // Mock intensity
    }
    
    private function getMockBotData() {
        return [
            ['address' => '0x1234...5678', 'total_profit' => 18.3, 'successful_mev' => 650, 'total_attempts' => 890, 'victims_count' => 234],
            ['address' => '0xabcd...efgh', 'total_profit' => 12.7, 'successful_mev' => 423, 'total_attempts' => 567, 'victims_count' => 156]
        ];
    }
    
    private function classifyBotStrategy($bot) {
        $sandwich_ratio = $bot['successful_mev'] / $bot['total_attempts'];
        if ($sandwich_ratio > 0.7) return 'SANDWICH_SPECIALIST';
        if ($sandwich_ratio > 0.5) return 'ARBITRAGE_BOT';
        return 'GENERALIST';
    }
    
    private function classifyBotThreat($bot) {
        $profit_per_hour = $bot['total_profit'] / 24;
        if ($profit_per_hour > 1) return 'CRITICAL';
        if ($profit_per_hour > 0.5) return 'HIGH';
        if ($profit_per_hour > 0.2) return 'MODERATE';
        return 'LOW';
    }
    
    private function getStrategyBreakdown($bots) {
        $breakdown = ['sandwich_attacks' => 0, 'arbitrage' => 0, 'liquidations' => 0, 'other' => 0];
        foreach ($bots as $bot) {
            switch ($bot['strategy']) {
                case 'SANDWICH_SPECIALIST':
                    $breakdown['sandwich_attacks']++;
                    break;
                case 'ARBITRAGE_BOT':
                    $breakdown['arbitrage']++;
                    break;
                default:
                    $breakdown['other']++;
            }
        }
        return $breakdown;
    }
    
    private function getProtocolData() {
        return [
            ['name' => 'Uniswap V2'],
            ['name' => 'Uniswap V3'],
            ['name' => 'Curve'],
            ['name' => 'CoW Protocol']
        ];
    }
    
    private function getProtocolMEVData($protocol) {
        // Mock protocol MEV data
        $mev_data = [
            'Uniswap V2' => ['attacks_24h' => 234, 'value_extracted_24h' => 87.3, 'users_affected_24h' => 156],
            'Uniswap V3' => ['attacks_24h' => 123, 'value_extracted_24h' => 45.6, 'users_affected_24h' => 89],
            'Curve' => ['attacks_24h' => 67, 'value_extracted_24h' => 23.4, 'users_affected_24h' => 45],
            'CoW Protocol' => ['attacks_24h' => 3, 'value_extracted_24h' => 0.2, 'users_affected_24h' => 2]
        ];
        
        return $mev_data[$protocol] ?? ['attacks_24h' => 0, 'value_extracted_24h' => 0, 'users_affected_24h' => 0];
    }
    
    private function checkMEVProtection($protocol) {
        $protected = ['CoW Protocol'];
        return in_array($protocol, $protected);
    }
    
    private function estimateActiveBots() {
        return 347; // Mock estimate
    }
    
    private function calculateTotalExtracted24h() {
        return 245.7; // Mock calculation
    }
    
    private function estimateNFTRarity($purchase_value) {
        return rand(30, 95); // Mock rarity score
    }
    
    private function generateSwapRecommendations($risk_score, $sandwich_probability) {
        $recommendations = [];
        
        if ($risk_score > 0.7) {
            $recommendations[] = "CRITICAL: Use Flashbots Protect or CoW Protocol";
        } elseif ($risk_score > 0.5) {
            $recommendations[] = "Set slippage <0.5%, consider splitting trade";
        }
        
        if ($sandwich_probability > 0.6) {
            $recommendations[] = "Consider limit orders instead of market orders";
        }
        
        return $recommendations;
    }
    
    /**
     * Main analysis function
     */
    public function analyze() {
        try {
            echo "🛡️ MEV Analyzer: Starting comprehensive MEV analysis...\n";
            
            // Analyze network MEV activity
            $network_analysis = $this->analyzeMEVActivity();
            
            // Analyze MEV bot landscape
            $bot_analysis = $this->analyzeMEVBots();
            
            // Assess protocol vulnerabilities
            $protocol_assessment = $this->assessProtocolVulnerability();
            
            // Calculate transaction risks
            $transaction_risks = [
                'eth_transfer' => $this->calculateTransactionRisk('eth_transfer'),
                'erc20_transfer' => $this->calculateTransactionRisk('erc20_transfer'),
                'dex_swap_small' => $this->calculateTransactionRisk('dex_swap', 100, 0.5),
                'dex_swap_large' => $this->calculateTransactionRisk('dex_swap', 5000, 2.0),
                'nft_purchase_rare' => $this->calculateTransactionRisk('nft_purchase', 1000),
                'liquidation' => $this->calculateTransactionRisk('liquidation')
            ];
            
            // Compile final intelligence
            $intelligence = [
                'mev_intelligence' => [
                    'network_status' => [
                        'mev_activity_level' => $network_analysis['activity_level'],
                        'total_extracted_24h' => $network_analysis['total_extracted_24h'],
                        'trend' => $network_analysis['recent_mev_trend'],
                        'avg_per_block' => round($network_analysis['avg_mev_per_block'], 3)
                    ],
                    
                    'bot_landscape' => [
                        'active_bots_24h' => $bot_analysis['active_bots_24h'],
                        'top_bot' => $bot_analysis['top_bots'][0] ?? null,
                        'strategy_breakdown' => $bot_analysis['strategy_breakdown']
                    ],
                    
                    'user_risk_assessment' => [
                        'transaction_risks' => $transaction_risks
                    ],
                    
                    'protocol_vulnerability' => $protocol_assessment,
                    
                    'recommendations' => [
                        'general' => [
                            "Current MEV activity {$network_analysis['activity_level']} - use protection for swaps >$1000",
                            "Consider private RPC for all DeFi interactions",
                            "Reduce slippage tolerance to 0.3-0.5%"
                        ],
                        'protection_services' => $this->getMEVProtectionServices()
                    ]
                ]
            ];
            
            echo "✅ MEV Analyzer: Analysis complete - {$network_analysis['activity_level']} activity detected\n";
            return $intelligence;
            
        } catch (Exception $e) {
            echo "❌ MEV Analyzer Error: " . $e->getMessage() . "\n";
            return ['error' => $e->getMessage()];
        }
    }
}

// Execute if run directly
if (basename(__FILE__) == basename($_SERVER['PHP_SELF'])) {
    echo "🛡️ Ethereum MEV Analyzer Starting Analysis...\n";
    
    $analyzer = new MEVAnalyzer();
    $intelligence = $analyzer->analyze();
    
    echo "\n" . json_encode($intelligence, JSON_PRETTY_PRINT) . "\n";
}
?>