<?php
/**
 * Ethereum Intelligence Oracle API
 * Purpose: Serve all transmutation intelligence to dashboard
 * Endpoints: /?transmutation=gas-prophet, mev-analyzer, staking-oracle, etc.
 * Output: JSON intelligence data for real-time display
 */

// CORS headers for cross-origin requests
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once 'transmutations/gas-prophet.php';
require_once 'transmutations/mev-analyzer.php';

/**
 * Load additional transmutations as they are built
 */
// require_once 'transmutations/staking-oracle.php';
// require_once 'transmutations/layer2-comparator.php';
// require_once 'transmutations/defi-opportunity-finder.php';
// require_once 'transmutations/executive-intelligence.php';

/**
 * Main Oracle API Handler
 */
class EthereumOracleAPI {
    private $available_transmutations = [
        'gas-prophet' => 'Gas Prophet - Gas price prediction and optimal timing',
        'mev-analyzer' => 'MEV Analyzer - MEV activity analysis and protection',
        'staking-oracle' => 'Staking Oracle - Staking strategy recommendations',
        'layer2-comparator' => 'L2 Comparator - Layer 2 network optimization',
        'defi-opportunity-finder' => 'DeFi Opportunity Finder - Yield optimization',
        'executive-intelligence' => 'Executive Intelligence - Strategic market synthesis'
    ];
    
    public function handleRequest() {
        try {
            $transmutation = $_GET['transmutation'] ?? 'gas-prophet';
            $action = $_GET['action'] ?? 'analyze';
            
            // Route request based on transmutation
            switch ($transmutation) {
                case 'gas-prophet':
                    $result = $this->handleGasProphet($action);
                    break;
                    
                case 'mev-analyzer':
                    $result = $this->handleMEVAnalyzer($action);
                    break;
                    
                case 'staking-oracle':
                    $result = $this->handleStakingOracle($action);
                    break;
                    
                case 'layer2-comparator':
                    $result = $this->handleLayer2Comparator($action);
                    break;
                    
                case 'defi-opportunity-finder':
                    $result = $this->handleDeFiOpportunityFinder($action);
                    break;
                    
                case 'executive-intelligence':
                    $result = $this->handleExecutiveIntelligence($action);
                    break;
                    
                case 'health':
                    $result = $this->getSystemHealth();
                    break;
                    
                case 'transmutations':
                    $result = $this->getAvailableTransmutations();
                    break;
                    
                default:
                    $result = [
                        'error' => 'Unknown transmutation',
                        'available' => array_keys($this->available_transmutations)
                    ];
            }
            
            // Add metadata to response
            $response = [
                'transmutation' => $transmutation,
                'action' => $action,
                'timestamp' => date('Y-m-d H:i:s'),
                'data' => $result
            ];
            
            http_response_code(200);
            echo json_encode($response, JSON_PRETTY_PRINT);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode([
                'error' => 'Internal server error',
                'message' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ], JSON_PRETTY_PRINT);
        }
    }
    
    /**
     * Handle Gas Prophet requests
     */
    private function handleGasProphet($action) {
        switch ($action) {
            case 'analyze':
                $prophet = new GasProphet();
                return $prophet->analyze();
                
            case 'current':
                return $this->getCurrentGasState();
                
            case 'predictions':
                return $this->getGasPredictions();
                
            case 'recommendations':
                return $this->getGasRecommendations();
                
            default:
                throw new Exception('Unknown Gas Prophet action: ' . $action);
        }
    }
    
    /**
     * Handle MEV Analyzer requests
     */
    private function handleMEVAnalyzer($action) {
        switch ($action) {
            case 'analyze':
                $analyzer = new MEVAnalyzer();
                return $analyzer->analyze();
                
            case 'current':
                return $this->getCurrentMEVState();
                
            case 'risks':
                return $this->getTransactionRisks();
                
            case 'protection':
                return $this->getProtectionServices();
                
            default:
                throw new Exception('Unknown MEV Analyzer action: ' . $action);
        }
    }
    
    /**
     * Handle Staking Oracle requests (placeholder)
     */
    private function handleStakingOracle($action) {
        // TODO: Implement when staking-oracle transmutation is built
        return [
            'error' => 'Staking Oracle transmutation not yet implemented',
            'status' => 'planned',
            'description' => 'Staking strategy recommendations and protocol comparisons'
        ];
    }
    
    /**
     * Handle L2 Comparator requests (placeholder)
     */
    private function handleLayer2Comparator($action) {
        // TODO: Implement when layer2-comparator transmutation is built
        return [
            'error' => 'Layer2 Comparator transmutation not yet implemented',
            'status' => 'planned',
            'description' => 'Layer 2 network optimization and recommendations'
        ];
    }
    
    /**
     * Handle DeFi Opportunity Finder requests (placeholder)
     */
    private function handleDeFiOpportunityFinder($action) {
        // TODO: Implement when defi-opportunity-finder transmutation is built
        return [
            'error' => 'DeFi Opportunity Finder transmutation not yet implemented',
            'status' => 'planned',
            'description' => 'Yield optimization and arbitrage opportunity detection'
        ];
    }
    
    /**
     * Handle Executive Intelligence requests (placeholder)
     */
    private function handleExecutiveIntelligence($action) {
        // TODO: Implement when executive-intelligence transmutation is built
        return [
            'error' => 'Executive Intelligence transmutation not yet implemented',
            'status' => 'planned',
            'description' => 'Strategic market synthesis combining all intelligence'
        ];
    }
    
    /**
     * Get system health status
     */
    private function getSystemHealth() {
        try {
            $db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                         "nvdaxcom_galliumhash_metabuilder_user", 
                         "32TbgulnE7YOd2G0e5");
            $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Check database connection
            $stmt = $db->query("SELECT 1");
            $db_status = 'healthy';
            
            // Check data freshness
            $stmt = $db->query("SELECT 
                MAX(block_number) as latest_block,
                COUNT(*) as total_records,
                MAX(timestamp) as latest_data
                FROM ethereum_intelligence");
            $data_status = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $latest_block = $data_status['latest_block'];
            $total_records = $data_status['total_records'];
            $latest_data = $data_status['latest_data'];
            
            // Calculate data freshness
            $data_age_minutes = $latest_data ? (time() - strtotime($latest_data)) / 60 : 999;
            
            if ($data_age_minutes > 60) {
                $data_status = 'stale';
            } elseif ($data_age_minutes > 10) {
                $data_status = 'aging';
            } else {
                $data_status = 'fresh';
            }
            
            return [
                'status' => 'operational',
                'database' => $db_status,
                'data_freshness' => [
                    'status' => $data_status,
                    'latest_block' => $latest_block,
                    'total_records' => $total_records,
                    'age_minutes' => round($data_age_minutes, 1),
                    'latest_data' => $latest_data
                ],
                'transmutations' => [
                    'gas-prophet' => $this->checkTransmutationHealth('gas-prophet'),
                    'mev-analyzer' => $this->checkTransmutationHealth('mev-analyzer'),
                    'staking-oracle' => 'planned',
                    'layer2-comparator' => 'planned',
                    'defi-opportunity-finder' => 'planned',
                    'executive-intelligence' => 'planned'
                ],
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }
    
    /**
     * Get available transmutations
     */
    private function getAvailableTransmutations() {
        return [
            'available' => $this->available_transmutations,
            'implemented' => ['gas-prophet', 'mev-analyzer'],
            'planned' => ['staking-oracle', 'layer2-comparator', 'defi-opportunity-finder', 'executive-intelligence'],
            'total' => count($this->available_transmutations),
            'implementation_status' => [
                'completed' => 2,
                'in_progress' => 0,
                'planned' => 4
            ]
        ];
    }
    
    /**
     * Check individual transmutation health
     */
    private function checkTransmutationHealth($transmutation) {
        try {
            $class_exists = class_exists(ucfirst(str_replace('-', '', $transmutation)));
            return $class_exists ? 'operational' : 'missing';
        } catch (Exception $e) {
            return 'error';
        }
    }
    
    /**
     * Get current gas state (quick endpoint)
     */
    private function getCurrentGasState() {
        try {
            $db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                         "nvdaxcom_galliumhash_metabuilder_user", 
                         "32TbgulnE7YOd2G0e5");
            $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            $stmt = $db->query("SELECT 
                base_fee, gas_price, timestamp, block_number
                FROM ethereum_intelligence 
                WHERE base_fee IS NOT NULL 
                ORDER BY block_number DESC 
                LIMIT 1");
            
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result) {
                return [
                    'current_gas' => round($result['base_fee'], 0),
                    'timestamp' => $result['timestamp'],
                    'block_number' => $result['block_number']
                ];
            }
            
            return ['error' => 'No gas data available'];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get gas predictions (quick endpoint)
     */
    private function getGasPredictions() {
        try {
            // Quick prediction without full analysis
            $gas_prophet = new GasProphet();
            $full_analysis = $gas_prophet->analyze();
            
            return $full_analysis['gas_intelligence']['predictions'] ?? [];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get gas recommendations (quick endpoint)
     */
    private function getGasRecommendations() {
        try {
            $gas_prophet = new GasProphet();
            $full_analysis = $gas_prophet->analyze();
            
            return $full_analysis['gas_intelligence']['recommendation'] ?? [];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get current MEV state (quick endpoint)
     */
    private function getCurrentMEVState() {
        try {
            $mev_analyzer = new MEVAnalyzer();
            $full_analysis = $mev_analyzer->analyze();
            
            return $full_analysis['mev_intelligence']['network_status'] ?? [];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get transaction risks (quick endpoint)
     */
    private function getTransactionRisks() {
        try {
            $mev_analyzer = new MEVAnalyzer();
            $full_analysis = $mev_analyzer->analyze();
            
            return $full_analysis['mev_intelligence']['user_risk_assessment']['transaction_risks'] ?? [];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get protection services (quick endpoint)
     */
    private function getProtectionServices() {
        try {
            $mev_analyzer = new MEVAnalyzer();
            $full_analysis = $mev_analyzer->analyze();
            
            return $full_analysis['mev_intelligence']['recommendations']['protection_services'] ?? [];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
}

/**
 * Error handling and logging
 */
function logError($message) {
    $log_entry = date('Y-m-d H:i:s') . " - ERROR: " . $message . "\n";
    error_log($log_entry, 3, '/tmp/ethereum-oracle-error.log');
}

// Initialize and handle the request
try {
    $oracle = new EthereumOracleAPI();
    $oracle->handleRequest();
    
} catch (Exception $e) {
    logError($e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Oracle API error',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT);
}
?>