<?php
/**
 * Ethereum GasCollector
 * Purpose: Collect comprehensive real-time gas data for intelligence analysis
 * Refresh Rate: Every 30 seconds (gas changes frequently)
 * API: Alchemy + Etherscan + Alternative sources
 * Storage: metachain database ethereum_intelligence table
 * cPanel Optimized: Runtime limits, error handling, caching
 */

class GasCollector {
    private $db;
    private $alchemy_api_key;
    private $etherscan_api_key;
    private $cache_duration = 30; // 30 seconds cache
    private $last_collection_time = 0;
    private $gas_history = [];
    
    public function __construct() {
        $this->db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                           "nvdaxcom_galliumhash_metabuilder_user", 
                           "32TbgulnE7YOd2G0e5");
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        $this->alchemy_api_key = "f3ZVHsvpZAMU-jDKbX_mk";
        $this->etherscan_api_key = "ZR2CAV375V5R3GCHWYVNZNBZZGTNZ5KH78";
        
        $this->loadGasHistory();
    }
    
    /**
     * Load recent gas history for pattern analysis
     */
    private function loadGasHistory() {
        try {
            $stmt = $this->db->prepare("SELECT gas_price, base_fee, priority_fee, timestamp 
                                       FROM ethereum_intelligence 
                                       WHERE timestamp > DATE_SUB(NOW(), INTERVAL 24 HOUR)
                                       ORDER BY timestamp DESC 
                                       LIMIT 720"); // Last 6 hours at 30-second intervals
            
            $stmt->execute();
            $this->gas_history = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo "GasCollector: Loaded " . count($this->gas_history) . " historical gas records\n";
        } catch (Exception $e) {
            echo "GasCollector Error loading history: " . $e->getMessage() . "\n";
            $this->gas_history = [];
        }
    }
    
    /**
     * Get real-time gas oracle data from multiple sources
     */
    private function getGasOracleData() {
        $gas_data = [];
        
        // Primary: Alchemy Gas Tracker
        try {
            $alchemy_data = $this->getAlchemyGasOracle();
            if ($alchemy_data) {
                $gas_data['alchemy'] = $alchemy_data;
            }
        } catch (Exception $e) {
            echo "Alchemy gas oracle failed: " . $e->getMessage() . "\n";
        }
        
        // Secondary: Etherscan Gas Tracker
        try {
            $etherscan_data = $this->getEtherscanGasOracle();
            if ($etherscan_data) {
                $gas_data['etherscan'] = $etherscan_data;
            }
        } catch (Exception $e) {
            echo "Etherscan gas oracle failed: " . $e->getMessage() . "\n";
        }
        
        // Tertiary: ETH Gas Station API
        try {
            $gasstation_data = $this->getGasStationData();
            if ($gasstation_data) {
                $gas_data['gasstation'] = $gasstation_data;
            }
        } catch (Exception $e) {
            echo "Gas Station API failed: " . $e->getMessage() . "\n";
        }
        
        return $gas_data;
    }
    
    /**
     * Get gas data from Alchemy Gas Tracker API
     */
    private function getAlchemyGasOracle() {
        $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
        $data = json_encode([
            "jsonrpc" => "2.0",
            "id" => 1,
            "method" => "alchemy_getGasPrices"
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if (isset($result['result'])) {
                return [
                    'source' => 'alchemy',
                    'timestamp' => time(),
                    'safe_low' => $result['result']['SafeLow']['maxFeePerGas'] ?? null,
                    'standard' => $result['result']['Standard']['maxFeePerGas'] ?? null,
                    'fast' => $result['result']['Fast']['maxFeePerGas'] ?? null,
                    'instant' => $result['result']['Instant']['maxFeePerGas'] ?? null,
                    'base_fee' => $result['result']['baseFee'] ?? null
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Get gas data from Etherscan
     */
    private function getEtherscanGasOracle() {
        $url = "https://api.etherscan.io/api?module=gastracker&action=gasoracle&apikey=" . $this->etherscan_api_key;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if ($result['status'] === '1' && isset($result['result'])) {
                return [
                    'source' => 'etherscan',
                    'timestamp' => time(),
                    'safe_low' => $result['result']['SafeLow']['MaxFeePerGas'] ?? null,
                    'standard' => $result['result']['ProposeGasPrice']['MaxFeePerGas'] ?? null,
                    'fast' => $result['result']['FastGasPrice']['MaxFeePerGas'] ?? null,
                    'instant' => $result['result']['FastGasPrice']['MaxFeePerGas'] * 1.2 ?? null,
                    'base_fee' => $result['result']['LastBlock']['BaseFee'] ?? null,
                    'priority_fee' => $result['result']['LastBlock']['PriorityFee'] ?? null
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Get gas data from ETH Gas Station (alternative source)
     */
    private function getGasStationData() {
        $url = "https://api.etherscan.io/api?module=gastracker&action=gasoracle&apikey=" . $this->etherscan_api_key;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if ($result['status'] === '1' && isset($result['result'])) {
                return [
                    'source' => 'gasstation',
                    'timestamp' => time(),
                    'safe_low' => $result['result']['SafeLow']['gasPrice'] ?? null,
                    'standard' => $result['result']['Standard']['gasPrice'] ?? null,
                    'fast' => $result['result']['Fast']['gasPrice'] ?? null,
                    'instant' => $result['result']['Fast']['gasPrice'] * 1.1 ?? null
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Get network congestion metrics
     */
    private function getNetworkCongestion() {
        $congestion_data = [];
        
        // Get pending transaction count from Alchemy
        try {
            $pending_data = $this->getPendingTransactions();
            if ($pending_data) {
                $congestion_data['pending_transactions'] = $pending_data['count'];
                $congestion_data['pending_block'] = $pending_data['block_number'];
            }
        } catch (Exception $e) {
            echo "Pending transactions API failed: " . $e->getMessage() . "\n";
        }
        
        // Get block utilization ratio
        try {
            $block_util = $this->getBlockUtilization();
            if ($block_util !== null) {
                $congestion_data['block_utilization'] = $block_util;
                $congestion_data['congestion_level'] = $this->classifyCongestion($block_util);
            }
        } catch (Exception $e) {
            echo "Block utilization check failed: " . $e->getMessage() . "\n";
        }
        
        // Estimate mempool size based on block times
        try {
            $mempool_size = $this->estimateMempoolSize();
            if ($mempool_size !== null) {
                $congestion_data['estimated_mempool_size'] = $mempool_size;
            }
        } catch (Exception $e) {
            echo "Mempool size estimation failed: " . $e->getMessage() . "\n";
        }
        
        return $congestion_data;
    }
    
    /**
     * Get pending transaction count from latest block
     */
    private function getPendingTransactions() {
        $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
        $data = json_encode([
            "jsonrpc" => "2.0",
            "id" => 1,
            "method" => "eth_getBlockTransactionCountByNumber",
            "params" => ["pending"]
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if (isset($result['result'])) {
                return [
                    'count' => hexdec($result['result']),
                    'block_number' => 'pending'
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Calculate block utilization ratio
     */
    private function getBlockUtilization() {
        $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
        $data = json_encode([
            "jsonrpc" => "2.0",
            "id" => 1,
            "method" => "eth_getBlockByNumber",
            "params" => ["latest", false] // Don't include transactions
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if (isset($result['result'])) {
                $gas_used = hexdec($result['result']['gasUsed']);
                $gas_limit = hexdec($result['result']['gasLimit']);
                
                return $gas_limit > 0 ? ($gas_used / $gas_limit) * 100 : 0;
            }
        }
        
        return null;
    }
    
    /**
     * Estimate mempool size based on recent activity
     */
    private function estimateMempoolSize() {
        // Simple estimation based on gas price patterns
        if (count($this->gas_history) < 10) {
            return null;
        }
        
        $recent_gas = array_slice($this->gas_history, 0, 10);
        $avg_gas = array_sum(array_column($recent_gas, 'gas_price')) / count($recent_gas);
        
        // Higher average gas = larger mempool
        if ($avg_gas > 100) return "large";      // >100 Gwei
        if ($avg_gas > 50) return "medium";      // >50 Gwei  
        if ($avg_gas > 20) return "small";       // >20 Gwei
        return "minimal";                        // <20 Gwei
    }
    
    /**
     * Classify congestion level
     */
    private function classifyCongestion($utilization) {
        if ($utilization > 95) return "critical";
        if ($utilization > 85) return "high";
        if ($utilization > 70) return "medium";
        if ($utilization > 50) return "low";
        return "minimal";
    }
    
    /**
     * Get gas price predictions using historical patterns
     */
    private function generateGasPredictions($current_gas_data) {
        if (count($this->gas_history) < 50) {
            return [
                '1h' => ['predicted' => null, 'confidence' => 0, 'reasoning' => 'Insufficient historical data'],
                '6h' => ['predicted' => null, 'confidence' => 0, 'reasoning' => 'Insufficient historical data'],
                '24h' => ['predicted' => null, 'confidence' => 0, 'reasoning' => 'Insufficient historical data']
            ];
        }
        
        $predictions = [];
        
        // 1-hour prediction (moving average with trend)
        $predictions['1h'] = $this->calculateShortTermPrediction(300); // 300 points = 2.5 hours
        
        // 6-hour prediction (daily pattern + trend)
        $predictions['6h'] = $this->calculateMediumTermPrediction(1800); // 1800 points = 15 hours
        
        // 24-hour prediction (weekly pattern + trend)
        $predictions['24h'] = $this->calculateLongTermPrediction(7200); // 7200 points = 60 hours
        
        return $predictions;
    }
    
    /**
     * Calculate short-term gas price prediction
     */
    private function calculateShortTermPrediction($points) {
        $recent_data = array_slice($this->gas_history, 0, min($points, count($this->gas_history)));
        
        if (count($recent_data) < 10) {
            return ['predicted' => null, 'confidence' => 0, 'reasoning' => 'Not enough data for prediction'];
        }
        
        $gas_prices = array_column($recent_data, 'gas_price');
        $current_price = $gas_prices[0];
        
        // Calculate moving averages
        $ma_5 = array_sum(array_slice($gas_prices, 0, 5)) / 5;
        $ma_15 = array_sum(array_slice($gas_prices, 0, 15)) / 15;
        $ma_30 = array_sum(array_slice($gas_prices, 0, 30)) / 30;
        
        // Calculate trend (simple linear regression)
        $trend = $this->calculateTrend($gas_prices, 10);
        
        // Predict next hour based on trend and moving averages
        $predicted_price = $current_price + ($trend * 60); // 60 minutes ahead
        
        // Confidence based on trend consistency and data availability
        $confidence = $this->calculatePredictionConfidence($recent_data, $trend);
        
        // Reasoning
        $reasoning = $this->generatePredictionReasoning($current_price, $predicted_price, $trend, $confidence);
        
        return [
            'predicted' => round($predicted_price, 2),
            'confidence' => round($confidence, 2),
            'reasoning' => $reasoning,
            'current' => $current_price,
            'trend' => round($trend, 3)
        ];
    }
    
    /**
     * Calculate medium-term gas price prediction (6 hours)
     */
    private function calculateMediumTermPrediction($points) {
        // Similar logic but for 6-hour timeframe
        return $this->calculatePredictionWithTimeframe($points, 360); // 6 hours = 360 minutes
    }
    
    /**
     * Calculate long-term gas price prediction (24 hours)
     */
    private function calculateLongTermPrediction($points) {
        // Similar logic but for 24-hour timeframe
        return $this->calculatePredictionWithTimeframe($points, 1440); // 24 hours = 1440 minutes
    }
    
    /**
     * General prediction calculation with timeframe
     */
    private function calculatePredictionWithTimeframe($points, $minutes_ahead) {
        $recent_data = array_slice($this->gas_history, 0, min($points, count($this->gas_history)));
        
        if (count($recent_data) < 20) {
            return ['predicted' => null, 'confidence' => 0, 'reasoning' => 'Insufficient historical data'];
        }
        
        $gas_prices = array_column($recent_data, 'gas_price');
        $current_price = $gas_prices[0];
        
        // Calculate multiple moving averages for pattern recognition
        $ma_short = array_sum(array_slice($gas_prices, 0, 10)) / 10;
        $ma_medium = array_sum(array_slice($gas_prices, 0, 30)) / 30;
        $ma_long = array_sum(array_slice($gas_prices, 0, 60)) / 60;
        
        // Calculate daily/weekly patterns
        $hourly_pattern = $this->calculateHourlyPattern();
        $current_hour = date('H', time());
        $pattern_adjustment = $hourly_pattern[$current_hour] ?? 1.0;
        
        // Calculate trend with longer timeframe
        $trend = $this->calculateTrend($gas_prices, min(60, count($gas_prices)));
        
        // Apply pattern adjustment to trend
        $pattern_adjusted_trend = $trend * $pattern_adjustment;
        
        // Predict price
        $predicted_price = $current_price + ($pattern_adjusted_trend * $minutes_ahead);
        
        // Confidence decreases with longer timeframes
        $base_confidence = $this->calculatePredictionConfidence($recent_data, $pattern_adjusted_trend);
        $time_decay = 1.0 - ($minutes_ahead / 1440); // Decay over 24 hours
        $confidence = $base_confidence * $time_decay;
        
        $reasoning = "Pattern-based prediction with {$current_hour}:00 hour adjustment";
        
        return [
            'predicted' => round($predicted_price, 2),
            'confidence' => round(max(0, $confidence), 2),
            'reasoning' => $reasoning,
            'current' => $current_price,
            'pattern_adjustment' => round($pattern_adjustment, 3),
            'trend' => round($pattern_adjusted_trend, 4)
        ];
    }
    
    /**
     * Calculate hourly gas price patterns
     */
    private function calculateHourlyPattern() {
        $hourly_averages = [];
        
        // Group gas prices by hour of day
        foreach ($this->gas_history as $record) {
            $hour = date('H', strtotime($record['timestamp']));
            $hourly_averages[$hour][] = $record['gas_price'];
        }
        
        // Calculate average for each hour and normalize
        $hourly_pattern = [];
        $overall_average = array_sum(array_column($this->gas_history, 'gas_price')) / count($this->gas_history);
        
        for ($hour = 0; $hour < 24; $hour++) {
            if (isset($hourly_averages[$hour]) && count($hourly_averages[$hour]) > 0) {
                $hour_avg = array_sum($hourly_averages[$hour]) / count($hourly_averages[$hour]);
                $hourly_pattern[$hour] = $overall_average > 0 ? $hour_avg / $overall_average : 1.0;
            } else {
                $hourly_pattern[$hour] = 1.0; // Default to neutral
            }
        }
        
        return $hourly_pattern;
    }
    
    /**
     * Calculate trend using linear regression
     */
    private function calculateTrend($data, $periods) {
        $recent_data = array_slice($data, 0, min($periods, count($data)));
        $count = count($recent_data);
        
        if ($count < 2) return 0;
        
        $sum_x = 0;
        $sum_y = 0;
        $sum_xy = 0;
        $sum_x2 = 0;
        
        for ($i = 0; $i < $count; $i++) {
            $x = $i;
            $y = $recent_data[$i];
            $sum_x += $x;
            $sum_y += $y;
            $sum_xy += $x * $y;
            $sum_x2 += $x * $x;
        }
        
        $n = $count;
        $denominator = ($n * $sum_x2 - $sum_x * $sum_x);
        
        if ($denominator == 0) return 0;
        
        // Slope = (n*sum_xy - sum_x*sum_y) / (n*sum_x2 - sum_x^2)
        $slope = ($n * $sum_xy - $sum_x * $sum_y) / $denominator;
        
        return $slope;
    }
    
    /**
     * Calculate prediction confidence score
     */
    private function calculatePredictionConfidence($data, $trend) {
        $base_confidence = min(count($data) / 100, 1.0); // Max confidence with 100+ data points
        
        // Adjust confidence based on trend consistency
        $volatility = $this->calculateVolatility($data);
        $trend_strength = 1.0 - min($volatility / 50, 1.0); // Lower volatility = higher confidence
        
        return $base_confidence * $trend_strength * 100; // Return as percentage
    }
    
    /**
     * Calculate price volatility
     */
    private function calculateVolatility($data) {
        if (count($data) < 2) return 0;
        
        $prices = array_column($data, 'gas_price');
        $mean = array_sum($prices) / count($prices);
        
        $variance = 0;
        foreach ($prices as $price) {
            $variance += pow($price - $mean, 2);
        }
        
        $standard_deviation = sqrt($variance / count($prices));
        return $standard_deviation;
    }
    
    /**
     * Generate prediction reasoning text
     */
    private function generatePredictionReasoning($current, $predicted, $trend, $confidence) {
        $change = $predicted - $current;
        $change_percent = ($change / $current) * 100;
        
        $direction = $change > 0 ? "increasing" : "decreasing";
        $strength = abs($change_percent) > 20 ? "significantly" : (abs($change_percent) > 5 ? "moderately" : "slightly");
        
        return "Gas prices expected to {$direction} {$strength} ({$change_percent:.1f}% change) with {$confidence:.1f}% confidence based on trend analysis";
    }
    
    /**
     * Detect upcoming events that might affect gas prices
     */
    private function detectGasEvents() {
        $events = [];
        $current_time = time();
        
        // NFT mint detection (simplified - would need more complex integration)
        // For now, we'll use gas spikes as event indicators
        
        if (count($this->gas_history) > 5) {
            $recent_gas = array_column(array_slice($this->gas_history, 0, 5), 'gas_price');
            $avg_recent = array_sum($recent_gas) / count($recent_gas);
            $previous_avg = array_sum(array_column(array_slice($this->gas_history, 5, 5), 'gas_price')) / 5;
            
            $spike_percent = (($avg_recent - $previous_avg) / $previous_avg) * 100;
            
            if ($spike_percent > 50) {
                $events[] = [
                    'type' => 'gas_spike',
                    'description' => "Gas price spike detected (+{$spike_percent:.1f}%)",
                    'impact' => 'high',
                    'timestamp' => time()
                ];
            }
        }
        
        // Protocol activity detection based on gas patterns
        $high_gas_threshold = 100; // Gwei
        $current_gas = $this->gas_history[0]['gas_price'] ?? 0;
        
        if ($current_gas > $high_gas_threshold) {
            $events[] = [
                'type' => 'high_network_activity',
                'description' => "High network activity detected (current gas: {$current_gas} Gwei)",
                'impact' => 'medium',
                'timestamp' => time()
            ];
        }
        
        return $events;
    }
    
    /**
     * Process and store all collected gas data
     */
    private function storeGasData($gas_oracle_data, $congestion_data, $predictions, $events) {
        try {
            // Aggregate gas data from multiple sources
            $aggregated_gas = $this->aggregateGasData($gas_oracle_data);
            
            // Check if we have valid data to store
            if (!$aggregated_gas || $aggregated_gas['standard'] <= 0) {
                echo "GasCollector: No valid gas data to store\n";
                return false;
            }
            
            // Get current block number for reference
            $current_block = $this->getCurrentBlockNumber();
            
            // Prepare data for storage
            $gas_price = $aggregated_gas['standard'];
            $base_fee = $aggregated_gas['base_fee'] ?? ($gas_price * 0.8);
            $priority_fee = $aggregated_gas['priority_fee'] ?? ($gas_price * 0.2);
            
            // Prepare congestion metrics
            $congestion_level = $congestion_data['congestion_level'] ?? 'unknown';
            $block_utilization = $congestion_data['block_utilization'] ?? 0;
            $pending_count = $congestion_data['pending_transactions'] ?? 0;
            
            // Prepare predictions JSON
            $predictions_json = json_encode([
                'predictions' => $predictions,
                'congestion' => $congestion_data,
                'events' => $events,
                'sources' => array_keys($gas_oracle_data),
                'collected_at' => time()
            ]);
            
            // Check if record for this block already exists
            if ($current_block) {
                $stmt = $this->db->prepare("SELECT id FROM ethereum_intelligence WHERE block_number = ?");
                $stmt->execute([$current_block]);
                
                if ($stmt->rowCount() > 0) {
                    // Update existing record
                    $sql = "UPDATE ethereum_intelligence SET 
                            gas_price = ?, base_fee = ?, priority_fee = ?,
                            block_number = ?, l2_comparisons = ?,
                            timestamp = NOW()
                            WHERE block_number = ?";
                } else {
                    // Insert new record
                    $sql = "INSERT INTO ethereum_intelligence (
                            gas_price, base_fee, priority_fee, block_number, l2_comparisons, timestamp
                        ) VALUES (?, ?, ?, ?, ?, NOW())";
                }
            } else {
                // Insert with timestamp-based record
                $sql = "INSERT INTO ethereum_intelligence (
                        gas_price, base_fee, priority_fee, l2_comparisons, timestamp
                    ) VALUES (?, ?, ?, ?, NOW())";
            }
            
            $stmt = $this->db->prepare($sql);
            $result = $stmt->execute([
                $gas_price,
                $base_fee,
                $priority_fee,
                $current_block,
                $predictions_json
            ]);
            
            if ($result) {
                echo "GasCollector: Stored gas data - Gas: {$gas_price} Gwei, Block: " . ($current_block ?? 'N/A') . "\n";
                
                // Update gas history
                $this->gas_history = array_merge([[
                    'gas_price' => $gas_price,
                    'base_fee' => $base_fee,
                    'priority_fee' => $priority_fee,
                    'timestamp' => date('Y-m-d H:i:s')
                ]], $this->gas_history);
                
                // Keep only last 720 records (6 hours)
                if (count($this->gas_history) > 720) {
                    $this->gas_history = array_slice($this->gas_history, 0, 720);
                }
                
                return true;
            }
            
        } catch (Exception $e) {
            echo "GasCollector Database Error: " . $e->getMessage() . "\n";
            return false;
        }
        
        return false;
    }
    
    /**
     * Aggregate gas data from multiple sources
     */
    private function aggregateGasData($gas_sources) {
        if (empty($gas_sources)) {
            return null;
        }
        
        $aggregated = [];
        $sources = array_keys($gas_sources);
        
        foreach (['safe_low', 'standard', 'fast', 'instant', 'base_fee', 'priority_fee'] as $field) {
            $values = [];
            foreach ($gas_sources as $source => $data) {
                if (isset($data[$field]) && $data[$field] !== null) {
                    $values[] = $data[$field];
                }
            }
            
            if (!empty($values)) {
                $aggregated[$field] = array_sum($values) / count($values);
            } else {
                $aggregated[$field] = null;
            }
        }
        
        return $aggregated;
    }
    
    /**
     * Get current block number for reference
     */
    private function getCurrentBlockNumber() {
        try {
            $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
            $data = json_encode([
                "jsonrpc" => "2.0",
                "id" => 1,
                "method" => "eth_blockNumber"
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($response && $http_code === 200) {
                $result = json_decode($response, true);
                if (isset($result['result'])) {
                    return hexdec($result['result']);
                }
            }
        } catch (Exception $e) {
            // Silently fail - block number is optional
        }
        
        return null;
    }
    
    /**
     * Main gas collection cycle
     */
    public function collect() {
        $start_time = time();
        $collections_successful = 0;
        $collections_failed = 0;
        
        echo "GasCollector: Starting collection cycle at " . date('Y-m-d H:i:s') . "\n";
        
        // Check if we need to collect (respect cache duration)
        $time_since_last = time() - $this->last_collection_time;
        if ($time_since_last < $this->cache_duration) {
            $remaining_cache = $this->cache_duration - $time_since_last;
            echo "GasCollector: Using cached data, next collection in {$remaining_cache}s\n";
            return [
                'status' => 'cached',
                'time_until_next' => $remaining_cache,
                'last_collection' => $this->last_collection_time
            ];
        }
        
        try {
            // Step 1: Collect gas oracle data from multiple sources
            echo "GasCollector: Collecting gas oracle data...\n";
            $gas_oracle_data = $this->getGasOracleData();
            
            if (empty($gas_oracle_data)) {
                throw new Exception("No gas oracle data available from any source");
            }
            
            echo "GasCollector: Gas oracle data collected from: " . implode(', ', array_keys($gas_oracle_data)) . "\n";
            
            // Step 2: Get network congestion metrics
            echo "GasCollector: Analyzing network congestion...\n";
            $congestion_data = $this->getNetworkCongestion();
            
            // Step 3: Generate gas price predictions
            echo "GasCollector: Generating gas price predictions...\n";
            $predictions = $this->generateGasPredictions($gas_oracle_data);
            
            // Step 4: Detect gas-affecting events
            echo "GasCollector: Detecting gas-affecting events...\n";
            $events = $this->detectGasEvents();
            
            // Step 5: Store all collected data
            echo "GasCollector: Storing gas intelligence data...\n";
            if ($this->storeGasData($gas_oracle_data, $congestion_data, $predictions, $events)) {
                $collections_successful++;
                $this->last_collection_time = time();
            } else {
                $collections_failed++;
            }
            
        } catch (Exception $e) {
            echo "GasCollector Critical Error: " . $e->getMessage() . "\n";
            $collections_failed++;
        }
        
        $duration = time() - $start_time;
        echo "GasCollector: Cycle complete - {$collections_successful} successful, {$collections_failed} failed in {$duration}s\n";
        
        return [
            'collections_successful' => $collections_successful,
            'collections_failed' => $collections_failed,
            'duration' => $duration,
            'gas_sources' => isset($gas_oracle_data) ? array_keys($gas_oracle_data) : [],
            'predictions_generated' => !empty($predictions),
            'events_detected' => count($events ?? []),
            'cache_duration' => $this->cache_duration,
            'last_collection' => $this->last_collection_time
        ];
    }
    
    /**
     * Get gas intelligence summary for dashboard
     */
    public function getGasSummary() {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    gas_price,
                    base_fee,
                    priority_fee,
                    l2_comparisons,
                    timestamp
                FROM ethereum_intelligence 
                WHERE timestamp > DATE_SUB(NOW(), INTERVAL 1 HOUR)
                ORDER BY timestamp DESC 
                LIMIT 10
            ");
            
            $stmt->execute();
            $recent_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($recent_data)) {
                return [
                    'status' => 'no_data',
                    'message' => 'No recent gas data available'
                ];
            }
            
            $latest = $recent_data[0];
            $parsed_comparisons = json_decode($latest['l2_comparisons'], true);
            
            return [
                'status' => 'success',
                'current_gas' => [
                    'gas_price' => $latest['gas_price'],
                    'base_fee' => $latest['base_fee'],
                    'priority_fee' => $latest['priority_fee']
                ],
                'predictions' => $parsed_comparisons['predictions'] ?? [],
                'congestion' => $parsed_comparisons['congestion'] ?? [],
                'events' => $parsed_comparisons['events'] ?? [],
                'sources' => $parsed_comparisons['sources'] ?? [],
                'last_updated' => $latest['timestamp'],
                'trend' => $this->calculateRecentTrend($recent_data)
            ];
            
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Calculate recent gas price trend
     */
    private function calculateRecentTrend($data) {
        if (count($data) < 5) return 'unknown';
        
        $prices = array_column($data, 'gas_price');
        $recent_5 = array_slice($prices, 0, 5);
        $older_5 = array_slice($prices, 5, 5);
        
        if (count($older_5) < 5) return 'insufficient_data';
        
        $recent_avg = array_sum($recent_5) / count($recent_5);
        $older_avg = array_sum($older_5) / count($older_5);
        
        $change_percent = (($recent_avg - $older_avg) / $older_avg) * 100;
        
        if ($change_percent > 5) return 'increasing';
        if ($change_percent < -5) return 'decreasing';
        return 'stable';
    }
}

// Execute if run directly
if (basename(__FILE__) == basename($_SERVER['PHP_SELF'])) {
    echo "⛽ Ethereum GasCollector Starting...\n";
    
    $collector = new GasCollector();
    $result = $collector->collect();
    
    echo "✅ GasCollector Results:\n";
    echo "   Status: {$result['status']}\n";
    echo "   Gas Sources: " . implode(', ', $result['gas_sources'] ?? []) . "\n";
    echo "   Predictions Generated: " . ($result['predictions_generated'] ? 'Yes' : 'No') . "\n";
    echo "   Events Detected: {$result['events_detected']}\n";
    echo "   Cache Duration: {$result['cache_duration']}s\n";
    
    if (isset($result['collections_successful'])) {
        echo "   Successful Collections: {$result['collections_successful']}\n";
        echo "   Failed Collections: {$result['collections_failed']}\n";
        echo "   Duration: {$result['duration']}s\n";
    }
}
?>