<?php
/**
 * Ethereum BlockCollector
 * Purpose: Collect every Ethereum block in real-time
 * Refresh Rate: Every 12 seconds (per block)
 * API: Alchemy + Etherscan (backup)
 * Storage: metachain database
 */

class BlockCollector {
    private $db;
    private $alchemy_api_key;
    private $etherscan_api_key;
    private $last_processed_block;
    
    public function __construct() {
        $this->db = new PDO("mysql:host=localhost;dbname=nvdaxcom_galliumhash_metachain;charset=utf8mb4", 
                           "nvdaxcom_galliumhash_metabuilder_user", 
                           "32TbgulnE7YOd2G0e5");
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        $this->alchemy_api_key = "f3ZVHsvpZAMU-jDKbX_mk";
        $this->etherscan_api_key = "ZR2CAV375V5R3GCHWYVNZNBZZGTNZ5KH78";
        
        $this->loadLastProcessedBlock();
    }
    
    /**
     * Get the last processed block from database
     */
    private function loadLastProcessedBlock() {
        try {
            $stmt = $this->db->query("SELECT MAX(block_number) as last_block FROM ethereum_intelligence");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $this->last_processed_block = $result['last_block'] ?? 0;
            
            if ($this->last_processed_block > 0) {
                echo "BlockCollector: Continuing from block " . $this->last_processed_block . "\n";
            } else {
                echo "BlockCollector: Starting fresh - fetching latest blocks\n";
                $this->last_processed_block = $this->getLatestBlockNumber() - 10; // Start 10 blocks behind
            }
        } catch (Exception $e) {
            echo "BlockCollector Error: " . $e->getMessage() . "\n";
            $this->last_processed_block = $this->getLatestBlockNumber() - 10;
        }
    }
    
    /**
     * Get latest block number from Alchemy
     */
    private function getLatestBlockNumber() {
        $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
        $data = json_encode([
            "jsonrpc" => "2.0",
            "id" => 1,
            "method" => "eth_blockNumber"
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if (isset($result['result'])) {
                return hexdec($result['result']);
            }
        }
        
        throw new Exception("Failed to get latest block number");
    }
    
    /**
     * Get full block data from Alchemy
     */
    private function getBlockData($block_number) {
        $url = "https://eth-mainnet.alchemyapi.io/v2/" . $this->alchemy_api_key;
        $data = json_encode([
            "jsonrpc" => "2.0",
            "id" => 1,
            "method" => "eth_getBlockByNumber",
            "params" => [decHex($block_number), true] // true = include transactions
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            return json_decode($response, true);
        }
        
        throw new Exception("Failed to get block data for block $block_number");
    }
    
    /**
     * Get block reward data from Etherscan
     */
    private function getBlockReward($block_number) {
        $url = "https://api.etherscan.io/api?module=block&action=getblockreward&blockno=" . $block_number . "&apikey=" . $this->etherscan_api_key;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response && $http_code === 200) {
            $result = json_decode($response, true);
            if ($result['status'] === '1' && isset($result['result'])) {
                return $result['result'];
            }
        }
        
        return null; // Etherscan data not available
    }
    
    /**
     * Process and store block data
     */
    private function processBlock($block_number, $block_data, $reward_data = null) {
        $block = $block_data['result'];
        
        // Calculate block metrics
        $transaction_count = count($block['transactions']);
        $gas_used = hexdec($block['gasUsed']);
        $gas_limit = hexdec($block['gasLimit']);
        $base_fee_per_gas = isset($block['baseFeePerGas']) ? hexdec($block['baseFeePerGas']) : 0;
        $priority_fee = $this->calculatePriorityFee($block);
        
        // Calculate ETH transferred in block
        $eth_transferred = $this->calculateETHTransferred($block['transactions']);
        
        // Total fees collected (base + priority)
        $total_fees = ($gas_used * $base_fee_per_gas) / 1e18 + ($gas_used * $priority_fee) / 1e18;
        
        // Block statistics
        $block_size = strlen(json_encode($block));
        
        // Validator/proposer info (post-merge)
        $block_producer = isset($block['miner']) ? $block['miner'] : null;
        
        // Gas price statistics
        $gas_prices = $this->extractGasPrices($block['transactions']);
        $avg_gas_price = count($gas_prices) > 0 ? array_sum($gas_prices) / count($gas_prices) : 0;
        $median_gas_price = $this->calculateMedian($gas_prices);
        
        try {
            // Check if block already exists
            $stmt = $this->db->prepare("SELECT id FROM ethereum_intelligence WHERE block_number = ?");
            $stmt->execute([$block_number]);
            
            if ($stmt->rowCount() > 0) {
                // Update existing record
                $sql = "UPDATE ethereum_intelligence SET 
                        gas_price = ?, base_fee = ?, priority_fee = ?,
                        block_size = ?, transaction_count = ?, eth_transferred = ?,
                        total_fees = ?, avg_gas_price = ?, median_gas_price = ?,
                        block_producer = ?, blob_gas_used = ?, excess_blob_gas = ?,
                        timestamp = FROM_UNIXTIME(?)
                        WHERE block_number = ?";
            } else {
                // Insert new record
                $sql = "INSERT INTO ethereum_intelligence (
                        block_number, gas_price, base_fee, priority_fee, block_size,
                        transaction_count, eth_transferred, total_fees, avg_gas_price,
                        median_gas_price, block_producer, blob_gas_used, excess_blob_gas, timestamp
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, FROM_UNIXTIME(?))";
            }
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                $base_fee_per_gas / 1e9, // Convert to Gwei
                $base_fee_per_gas / 1e9,
                $priority_fee / 1e9,
                $block_size,
                $transaction_count,
                $eth_transferred,
                $total_fees,
                $avg_gas_price / 1e9, // Convert to Gwei
                $median_gas_price / 1e9, // Convert to Gwei
                $block_producer,
                isset($block['blobGasUsed']) ? hexdec($block['blobGasUsed']) : 0,
                isset($block['excessBlobGas']) ? hexdec($block['excessBlobGas']) : 0,
                hexdec($block['timestamp']),
                $block_number
            ]);
            
            return true;
        } catch (Exception $e) {
            echo "Database Error: " . $e->getMessage() . "\n";
            return false;
        }
    }
    
    /**
     * Calculate priority fee from transactions
     */
    private function calculatePriorityFee($block) {
        $base_fee = isset($block['baseFeePerGas']) ? hexdec($block['baseFeePerGas']) : 0;
        $transactions = $block['transactions'];
        $priority_fees = [];
        
        foreach ($transactions as $tx) {
            if (isset($tx['maxPriorityFeePerGas'])) {
                $priority_fees[] = hexdec($tx['maxPriorityFeePerGas']);
            }
        }
        
        return count($priority_fees) > 0 ? array_sum($priority_fees) / count($priority_fees) : 0;
    }
    
    /**
     * Calculate total ETH transferred in block
     */
    private function calculateETHTransferred($transactions) {
        $total_eth = 0;
        
        foreach ($transactions as $tx) {
            // Calculate value transferred in this transaction
            $value = isset($tx['value']) ? hexdec($tx['value']) : 0;
            $gas_used = isset($tx['gas']) ? hexdec($tx['gas']) : 21000;
            $gas_price = isset($tx['gasPrice']) ? hexdec($tx['gasPrice']) : 0;
            
            // Value in ETH (not including gas fees)
            $total_eth += $value / 1e18;
        }
        
        return $total_eth;
    }
    
    /**
     * Extract gas prices from transactions for analysis
     */
    private function extractGasPrices($transactions) {
        $gas_prices = [];
        
        foreach ($transactions as $tx) {
            if (isset($tx['gasPrice'])) {
                $gas_prices[] = hexdec($tx['gasPrice']);
            }
        }
        
        return $gas_prices;
    }
    
    /**
     * Calculate median from array
     */
    private function calculateMedian($numbers) {
        if (empty($numbers)) return 0;
        
        sort($numbers);
        $count = count($numbers);
        $middle = floor($count / 2);
        
        if ($count % 2 == 0) {
            return ($numbers[$middle - 1] + $numbers[$middle]) / 2;
        } else {
            return $numbers[$middle];
        }
    }
    
    /**
     * Main collection loop
     */
    public function collect() {
        $start_time = time();
        $blocks_collected = 0;
        $errors = 0;
        
        echo "BlockCollector: Starting collection cycle at " . date('Y-m-d H:i:s') . "\n";
        
        try {
            $latest_block = $this->getLatestBlockNumber();
            
            for ($block_num = $this->last_processed_block + 1; $block_num <= $latest_block; $block_num++) {
                try {
                    echo "Processing block $block_num...\n";
                    
                    // Get block data from Alchemy
                    $block_data = $this->getBlockData($block_num);
                    
                    if ($block_data && isset($block_data['result'])) {
                        // Get reward data from Etherscan (backup/supplementary)
                        $reward_data = $this->getBlockReward($block_num);
                        
                        // Process and store block
                        if ($this->processBlock($block_num, $block_data, $reward_data)) {
                            $blocks_collected++;
                            $this->last_processed_block = $block_num;
                            
                            // Progress update every 100 blocks
                            if ($blocks_collected % 100 == 0) {
                                echo "BlockCollector: Processed $blocks_collected blocks so far...\n";
                            }
                        } else {
                            $errors++;
                        }
                        
                        // Small delay to prevent rate limiting
                        usleep(50000); // 50ms delay
                    } else {
                        echo "BlockCollector: Failed to get data for block $block_num\n";
                        $errors++;
                    }
                    
                } catch (Exception $e) {
                    echo "BlockCollector Error on block $block_num: " . $e->getMessage() . "\n";
                    $errors++;
                    
                    // If consecutive errors, break to prevent infinite loop
                    if ($errors > 5) {
                        echo "BlockCollector: Too many errors, breaking collection cycle\n";
                        break;
                    }
                }
                
                // Runtime limit check (prevent long-running scripts)
                if (time() - $start_time > 300) { // 5 minutes max
                    echo "BlockCollector: Runtime limit reached\n";
                    break;
                }
            }
            
        } catch (Exception $e) {
            echo "BlockCollector Critical Error: " . $e->getMessage() . "\n";
        }
        
        $duration = time() - $start_time;
        echo "BlockCollector: Cycle complete - {$blocks_collected} blocks, {$errors} errors in {$duration}s\n";
        
        return [
            'blocks_collected' => $blocks_collected,
            'errors' => $errors,
            'duration' => $duration,
            'last_processed' => $this->last_processed_block
        ];
    }
}

// Helper function to convert decimal to hex
function decHex($dec) {
    return '0x' . dechex($dec);
}

// Execute if run directly
if (basename(__FILE__) == basename($_SERVER['PHP_SELF'])) {
    echo "🚀 Ethereum BlockCollector Starting...\n";
    
    $collector = new BlockCollector();
    $result = $collector->collect();
    
    echo "✅ BlockCollector Results:\n";
    echo "   Blocks Collected: {$result['blocks_collected']}\n";
    echo "   Errors: {$result['errors']}\n";
    echo "   Duration: {$result['duration']}s\n";
    echo "   Last Processed: {$result['last_processed']}\n";
}
?>