-- MetaChain Ethereum Intelligence Database Schema
-- GalliumHash MetaChain - Standalone ETH Analytics
-- Database: nvdaxcom_galliumhash_metachain

-- Create database if it doesn't exist
CREATE DATABASE IF NOT EXISTS nvdaxcom_galliumhash_metachain;
USE nvdaxcom_galliumhash_metachain;

-- =============================================================================
-- GAS TRACKING TABLES
-- =============================================================================

-- Store historical gas price data
CREATE TABLE eth_gas_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    timestamp BIGINT NOT NULL,
    gas_price JSON NOT NULL,
    recommended JSON,
    block_number INT,
    base_fee BIGINT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_timestamp (timestamp),
    INDEX idx_created_at (created_at)
);

-- Store gas price predictions
CREATE TABLE eth_gas_predictions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    predicted_gas JSON NOT NULL,
    confidence DECIMAL(5,2),
    prediction_type ENUM('oracle', 'ai', 'historical') NOT NULL,
    blocks_ahead INT NOT NULL,
    actual_gas JSON,
    accuracy DECIMAL(5,2),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_timestamp (timestamp),
    INDEX idx_prediction_type (prediction_type)
);

-- =============================================================================
-- MEV TRACKING TABLES
-- =============================================================================

-- Store MEV attack data
CREATE TABLE eth_mev_attacks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    attack_type ENUM('sandwich', 'front_run', 'arbitrage', 'liquidation') NOT NULL,
    victim_tx_hash VARCHAR(66),
    attacker_address VARCHAR(42),
    victim_address VARCHAR(42),
    dex_name VARCHAR(50),
    token_in VARCHAR(42),
    token_out VARCHAR(42),
    amount_in DECIMAL(36,18),
    profit_usd DECIMAL(18,2),
    gas_used INT,
    gas_price BIGINT,
    block_number INT,
    timestamp BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_attack_type (attack_type),
    INDEX idx_block_number (block_number),
    INDEX idx_timestamp (timestamp),
    INDEX idx_dex_name (dex_name)
);

-- Store MEV bot tracking
CREATE TABLE eth_mev_bots (
    id INT AUTO_INCREMENT PRIMARY KEY,
    bot_address VARCHAR(42) NOT NULL,
    bot_name VARCHAR(100),
    strategy_type ENUM('sandwich', 'arbitrage', 'liquidation', 'mev_relay') NOT NULL,
    total_profit_usd DECIMAL(18,2) DEFAULT 0,
    attack_count INT DEFAULT 0,
    first_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    is_active BOOLEAN DEFAULT TRUE,
    INDEX idx_bot_address (bot_address),
    INDEX idx_strategy_type (strategy_type),
    INDEX idx_is_active (is_active)
);

-- Store arbitrage opportunities
CREATE TABLE eth_arbitrage_opportunities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    dex_a VARCHAR(50),
    dex_b VARCHAR(50),
    token_a VARCHAR(42),
    token_b VARCHAR(42),
    price_a DECIMAL(36,18),
    price_b DECIMAL(36,18),
    spread_percentage DECIMAL(10,4),
    volume_usd DECIMAL(18,2),
    profit_potential_usd DECIMAL(18,2),
    block_number INT,
    timestamp BIGINT NOT NULL,
    resolved_at TIMESTAMP NULL,
    executed BOOLEAN DEFAULT FALSE,
    INDEX idx_dex_pair (dex_a, dex_b),
    INDEX idx_timestamp (timestamp),
    INDEX idx_block_number (block_number)
);

-- =============================================================================
-- STAKING TRACKING TABLES
-- =============================================================================

-- Store staking pool data
CREATE TABLE eth_staking_pools (
    id INT AUTO_INCREMENT PRIMARY KEY,
    pool_address VARCHAR(42) NOT NULL,
    pool_name VARCHAR(100),
    total_staked DECIMAL(36,18) NOT NULL,
    validator_count INT DEFAULT 0,
    current_apr DECIMAL(10,4),
    projected_apy DECIMAL(10,4),
    min_deposit DECIMAL(36,18),
    fee_percentage DECIMAL(10,4),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_pool_address (pool_address),
    INDEX idx_is_active (is_active)
);

-- Store individual staking positions
CREATE TABLE eth_staking_positions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    staker_address VARCHAR(42) NOT NULL,
    pool_id INT,
    amount_staked DECIMAL(36,18) NOT NULL,
    reward_earned DECIMAL(36,18) DEFAULT 0,
    unlock_date BIGINT,
    status ENUM('active', 'unlocking', 'withdrawn') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_staker_address (staker_address),
    INDEX idx_pool_id (pool_id),
    INDEX idx_status (status),
    FOREIGN KEY (pool_id) REFERENCES eth_staking_pools(id)
);

-- Store validator performance data
CREATE TABLE eth_validators (
    id INT AUTO_INCREMENT PRIMARY KEY,
    validator_id BIGINT NOT NULL,
    validator_address VARCHAR(42),
    commission_rate DECIMAL(10,4),
    performance_score DECIMAL(10,4),
    slashing_events INT DEFAULT 0,
    uptime_percentage DECIMAL(10,4),
    stake_amount DECIMAL(36,18),
    is_active BOOLEAN DEFAULT TRUE,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_validator_id (validator_id),
    INDEX idx_validator_address (validator_address),
    INDEX idx_is_active (is_active)
);

-- =============================================================================
-- L2 NETWORK TABLES
-- =============================================================================

-- Store L2 network metrics
CREATE TABLE eth_l2_networks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    network_name VARCHAR(50) NOT NULL,
    network_id INT NOT NULL,
    rpc_endpoint VARCHAR(255),
    bridge_address VARCHAR(42),
    gas_token VARCHAR(10) DEFAULT 'ETH',
    current_tps DECIMAL(10,2),
    max_tps DECIMAL(10,2),
    avg_block_time DECIMAL(8,2),
    total_volume_24h DECIMAL(24,2),
    total_transactions_24h BIGINT,
    active_contracts BIGINT,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_network_name (network_name),
    INDEX idx_network_id (network_id)
);

-- Store L2 bridge operations
CREATE TABLE eth_bridge_operations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    operation_type ENUM('deposit', 'withdrawal') NOT NULL,
    l2_network VARCHAR(50) NOT NULL,
    user_address VARCHAR(42) NOT NULL,
    amount DECIMAL(36,18) NOT NULL,
    token_address VARCHAR(42),
    tx_hash VARCHAR(66),
    status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
    gas_used INT,
    fee_paid DECIMAL(36,18),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    INDEX idx_user_address (user_address),
    INDEX idx_l2_network (l2_network),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
);

-- =============================================================================
-- SECURITY & ALERTS TABLES
-- =============================================================================

-- Store security alerts
CREATE TABLE eth_security_alerts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    alert_type ENUM('mev_bot', 'gas_spike', 'contract_hack', 'large_transfer', 'suspicious_activity') NOT NULL,
    severity ENUM('critical', 'warning', 'info') NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    affected_addresses JSON,
    transaction_hashes JSON,
    metadata JSON,
    is_resolved BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP NULL,
    INDEX idx_alert_type (alert_type),
    INDEX idx_severity (severity),
    INDEX idx_is_resolved (is_resolved),
    INDEX idx_created_at (created_at)
);

-- =============================================================================
-- PRICE & MARKET DATA TABLES
-- =============================================================================

-- Store price data
CREATE TABLE eth_price_data (
    id INT AUTO_INCREMENT PRIMARY KEY,
    token_address VARCHAR(42) DEFAULT 'ETH',
    exchange VARCHAR(100) NOT NULL,
    price_usd DECIMAL(18,8) NOT NULL,
    volume_24h DECIMAL(24,2),
    market_cap BIGINT,
    price_change_24h DECIMAL(10,4),
    timestamp BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_token_address (token_address),
    INDEX idx_exchange (exchange),
    INDEX idx_timestamp (timestamp)
);

-- Store network metrics
CREATE TABLE eth_network_metrics (
    id INT AUTO_INCREMENT PRIMARY KEY,
    block_number INT NOT NULL,
    block_time DECIMAL(8,2),
    gas_used BIGINT,
    gas_limit BIGINT,
    base_fee_per_gas BIGINT,
    priority_fee_per_gas BIGINT,
    total_difficulty BIGINT,
    hashrate_soll_per_sec BIGINT,
    transaction_count INT,
    uncle_count INT,
    timestamp BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_block_number (block_number),
    INDEX idx_timestamp (timestamp)
);

-- =============================================================================
-- USER INTERACTIONS TABLES
-- =============================================================================

-- Store user predictions for universal mining
CREATE TABLE eth_user_predictions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id VARCHAR(100) NOT NULL,
    prediction_type ENUM('gas_price', 'block_time', 'mev_activity', 'staking_apy') NOT NULL,
    prediction_question TEXT NOT NULL,
    predicted_value DECIMAL(20,8),
    prediction_boolean BOOLEAN,
    stake_amount DECIMAL(18,2) DEFAULT 0,
    tokens_earned DECIMAL(18,2) DEFAULT 0,
    is_correct BOOLEAN,
    payout_amount DECIMAL(18,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_prediction_type (prediction_type),
    INDEX idx_created_at (created_at)
);

-- Store user alerts
CREATE TABLE eth_user_alerts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id VARCHAR(100) NOT NULL,
    alert_type ENUM('gas_price', 'mev_attack', 'staking_event', 'l2_activity') NOT NULL,
    condition_type ENUM('above', 'below', 'equals', 'change') NOT NULL,
    threshold_value DECIMAL(20,8),
    is_active BOOLEAN DEFAULT TRUE,
    notification_method ENUM('email', 'webhook', 'sms') DEFAULT 'email',
    last_triggered TIMESTAMP NULL,
    trigger_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_alert_type (alert_type),
    INDEX idx_is_active (is_active)
);

-- =============================================================================
-- INITIAL DATA SETUP
-- =============================================================================

-- Insert default L2 networks
INSERT INTO eth_l2_networks (network_name, network_id, rpc_endpoint) VALUES
('Arbitrum', 42161, 'https://arb1.arbitrum.io/rpc'),
('Optimism', 10, 'https://mainnet.optimism.io'),
('Base', 8453, 'https://mainnet.base.org'),
('Polygon', 137, 'https://polygon-rpc.com');

-- Insert default staking pools
INSERT INTO eth_staking_pools (pool_address, pool_name, min_deposit, fee_percentage) VALUES
('0xae7ab96520DE3A18E5e111B5EaAb095312D7fE84', 'Lido', 0.001, 0.05),
('0x8d72377b8ccb28F2D31C7a12a8e41eCe48E8FcC9', 'Rocket Pool', 0.01, 0.15),
('0x0034E98B4A10423Ded9E0a5B92a3c168Acc54d39', 'StakeWise V3', 0.001, 0.02);

-- Create initial security alert
INSERT INTO eth_security_alerts (alert_type, severity, title, message) VALUES
('suspicious_activity', 'info', 'Ethereum Intelligence Hub Initialized', 'Welcome to the Ethereum Intelligence Center! All monitoring systems are now active.');

-- =============================================================================
-- PERFORMANCE INDEXES
-- =============================================================================

-- Additional performance indexes for common queries
CREATE INDEX idx_mev_attacks_timestamp_block ON eth_mev_attacks (timestamp, block_number);
CREATE INDEX idx_gas_history_timestamp_block ON eth_gas_history (timestamp, block_number);
CREATE INDEX idx_price_data_token_timestamp ON eth_price_data (token_address, timestamp);
CREATE INDEX idx_network_metrics_timestamp ON eth_network_metrics (timestamp);

-- =============================================================================
-- ARCHIVE OLD DATA PROCEDURES
-- =============================================================================

DELIMITER //

CREATE PROCEDURE ArchiveOldData()
BEGIN
    -- Archive data older than 30 days
    -- Gas history older than 30 days
    DELETE FROM eth_gas_history WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY);
    
    -- MEV attacks older than 30 days
    DELETE FROM eth_mev_attacks WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY);
    
    -- Arbitrage opportunities older than 7 days (more frequent cleanup)
    DELETE FROM eth_arbitrage_opportunities WHERE created_at < DATE_SUB(NOW(), INTERVAL 7 DAY);
    
    -- Security alerts older than 90 days
    DELETE FROM eth_security_alerts WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY) AND is_resolved = TRUE;
    
END //

DELIMITER ;

-- =============================================================================
-- END OF SCHEMA
-- =============================================================================

-- Database is ready for the Ethereum Intelligence Hub!
-- Tables are designed for high-performance analytics and real-time monitoring