<?php
/**
 * Ethereum Intelligence API - Backend
 * GalliumHash MetaChain - Standalone ETH Analytics
 * 
 * Handles all Ethereum-specific data processing and API endpoints
 */

// CORS headers for cross-domain requests
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Database configuration for Ethereum Intelligence
$db_config = [
    'host' => 'localhost',
    'dbname' => 'nvdaxcom_galliumhash_metachain',
    'username' => 'nvdaxcom_galliumhash_metabuilder_user',
    'password' => '32TbgulnE7YOd2G0e5',
    'charset' => 'utf8mb4'
];

class EthereumIntelligenceAPI {
    private $db;
    
    public function __construct() {
        $this->connectDatabase();
    }
    
    private function connectDatabase() {
        try {
            $dsn = "mysql:host={$GLOBALS['db_config']['host']};dbname={$GLOBALS['db_config']['dbname']};charset={$GLOBALS['db_config']['charset']}";
            $this->db = new PDO($dsn, $GLOBALS['db_config']['username'], $GLOBALS['db_config']['password']);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            $this->db = null;
        }
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = $_GET['endpoint'] ?? '';
        
        try {
            switch ($method) {
                case 'GET':
                    return $this->handleGet($path);
                case 'POST':
                    return $this->handlePost($path);
                default:
                    return $this->errorResponse('Method not allowed', 405);
            }
        } catch (Exception $e) {
            error_log("API Error: " . $e->getMessage());
            return $this->errorResponse('Internal server error', 500);
        }
    }
    
    private function handleGet($endpoint) {
        switch ($endpoint) {
            case 'gas':
                return $this->getGasData();
            case 'mev':
                return $this->getMEVData();
            case 'staking':
                return $this->getStakingData();
            case 'l2':
                return $this->getL2Data();
            case 'network':
                return $this->getNetworkData();
            case 'alerts':
                return $this->getSecurityAlerts();
            case 'prices':
                return $this->getPriceData();
            case 'validators':
                return $this->getValidatorData();
            case 'blocks':
                return $this->getRecentBlocks();
            case 'mempool':
                return $this->getMempoolData();
            default:
                return $this->errorResponse('Endpoint not found', 404);
        }
    }
    
    private function handlePost($endpoint) {
        $input = json_decode(file_get_contents('php://input'), true);
        
        switch ($endpoint) {
            case 'prediction':
                return $this->createPrediction($input);
            case 'alert':
                return $this->createAlert($input);
            case 'feedback':
                return $this->submitFeedback($input);
            default:
                return $this->errorResponse('Endpoint not found', 404);
        }
    }
    
    private function getGasData() {
        // Get current gas prices from multiple sources
        $gasData = [
            'timestamp' => time(),
            'gas_price' => $this->fetchGasPrice(),
            'recommended' => $this->getGasRecommendation(),
            'history' => $this->getGasHistory(24), // Last 24 hours
            'oracle' => [
                'safe' => $this->getSafeGasPrice(),
                'standard' => $this->getStandardGasPrice(),
                'fast' => $this->getFastGasPrice()
            ]
        ];
        
        // Cache in database
        $this->cacheGasData($gasData);
        
        return $this->successResponse($gasData);
    }
    
    private function getMEVData() {
        // Get MEV-related statistics and alerts
        $mevData = [
            'timestamp' => time(),
            'sandwich_attacks' => $this->getSandwichAttackData(),
            'arbitrage' => $this->getArbitrageData(),
            'front_running' => $this->getFrontRunningData(),
            'priority_gas_auction' => $this->getPriorityGasData(),
            'mev_bots' => $this->getActiveMEVBots(),
            'stats' => [
                'total_attacks_24h' => $this->getAttackCount(24),
                'total_loss_24h' => $this->getTotalLoss(24),
                'active_bots' => $this->getBotCount()
            ]
        ];
        
        return $this->successResponse($mevData);
    }
    
    private function getStakingData() {
        // Get Ethereum staking information
        $stakingData = [
            'timestamp' => time(),
            'total_staked' => $this->getTotalStaked(),
            'validators' => [
                'active' => $this->getActiveValidatorCount(),
                'pending' => $this->getPendingValidatorCount(),
                'slashed' => $this->getSlashedValidatorCount()
            ],
            'rewards' => [
                'current_apr' => $this->getCurrentAPR(),
                'projected_apy' => $this->getProjectedAPY(),
                'next_payout' => $this->getNextPayout()
            ],
            'withdrawal_queue' => [
                'pending' => $this->getWithdrawalQueueLength(),
                'wait_time' => $this->getWithdrawalWaitTime()
            ]
        ];
        
        return $this->successResponse($stakingData);
    }
    
    private function getL2Data() {
        // Get Layer 2 network data
        $l2Data = [
            'timestamp' => time(),
            'networks' => [
                'arbitrum' => [
                    'name' => 'Arbitrum One',
                    'gas_price' => $this->getArbitrumGasPrice(),
                    'tps' => $this->getArbitrumTPS(),
                    'volume_24h' => $this->getArbitrumVolume(),
                    'status' => 'healthy'
                ],
                'optimism' => [
                    'name' => 'Optimism',
                    'gas_price' => $this->getOptimismGasPrice(),
                    'tps' => $this->getOptimismTPS(),
                    'volume_24h' => $this->getOptimismVolume(),
                    'status' => 'healthy'
                ],
                'base' => [
                    'name' => 'Base',
                    'gas_price' => $this->getBaseGasPrice(),
                    'tps' => $this->getBaseTPS(),
                    'volume_24h' => $this->getBaseVolume(),
                    'status' => 'healthy'
                ]
            ],
            'bridge_operations' => $this->getBridgeOperationData()
        ];
        
        return $this->successResponse($l2Data);
    }
    
    private function getNetworkData() {
        // Get general network statistics
        $networkData = [
            'timestamp' => time(),
            'block_number' => $this->getCurrentBlockNumber(),
            'block_time' => $this->getAverageBlockTime(),
            'difficulty' => $this->getCurrentDifficulty(),
            'hashrate' => $this->getNetworkHashrate(),
            'gas_used' => $this->getGasUsed24h(),
            'transactions' => [
                'total_24h' => $this->getTransactionCount24h(),
                'avg_per_block' => $this->getAverageTransactionsPerBlock(),
                'pending' => $this->getPendingTransactionCount()
            ],
            'fees' => [
                'total_fees_24h' => $this->getTotalFees24h(),
                'avg_fee' => $this->getAverageFee(),
                'base_fee' => $this->getCurrentBaseFee()
            ]
        ];
        
        return $this->successResponse($networkData);
    }
    
    private function getSecurityAlerts() {
        // Get recent security alerts and warnings
        $alerts = $this->db ? $this->getAlertsFromDatabase() : $this->generateMockAlerts();
        
        $alertData = [
            'timestamp' => time(),
            'alerts' => $alerts,
            'severity_levels' => [
                'critical' => count(array_filter($alerts, fn($a) => $a['severity'] === 'critical')),
                'warning' => count(array_filter($alerts, fn($a) => $a['severity'] === 'warning')),
                'info' => count(array_filter($alerts, fn($a) => $a['severity'] === 'info'))
            ]
        ];
        
        return $this->successResponse($alertData);
    }
    
    // Database query methods (placeholder implementations)
    private function fetchGasPrice() {
        // In production, this would fetch from multiple sources:
        // - Etherscan API
        // - Alchemy API
        // - Infura API
        // - Custom Ethereum nodes
        
        $gasPrices = [
            'safe' => 20 + rand(0, 10),
            'standard' => 15 + rand(0, 10),
            'fast' => 10 + rand(0, 10)
        ];
        
        return $gasPrices;
    }
    
    private function getGasRecommendation() {
        // AI-powered gas price recommendation
        $currentGas = $this->fetchGasPrice();
        $history = $this->getGasHistory(6); // Last 6 hours
        
        if (count($history) < 5) {
            return 'Use Safe Gas (30-50 gwei)';
        }
        
        $trend = $this->calculateGasTrend($history);
        
        switch ($trend) {
            case 'rising':
                return 'Use Fast Gas to avoid delays';
            case 'falling':
                return 'Wait for better rates';
            default:
                return 'Standard Gas price looks good';
        }
    }
    
    private function calculateGasTrend($history) {
        if (count($history) < 3) return 'stable';
        
        $recent = array_slice($history, -3);
        $older = array_slice($history, -6, 3);
        
        $recentAvg = array_sum(array_map(fn($h) => $h['gas_price'], $recent)) / 3;
        $olderAvg = array_sum(array_map(fn($h) => $h['gas_price'], $older)) / 3;
        
        if ($recentAvg > $olderAvg * 1.1) return 'rising';
        if ($recentAvg < $olderAvg * 0.9) return 'falling';
        return 'stable';
    }
    
    private function getSandwichAttackData() {
        // Get sandwich attack statistics
        return [
            'attacks_last_hour' => rand(50, 200),
            'avg_loss' => rand(500, 2000),
            'total_volume' => rand(1000000, 10000000),
            'most_affected_dexes' => [
                'Uniswap V3' => rand(40, 60),
                'SushiSwap' => rand(15, 25),
                '1inch' => rand(10, 20)
            ]
        ];
    }
    
    private function getArbitrageData() {
        // Get arbitrage opportunity data
        return [
            'opportunities_count' => rand(20, 100),
            'total_volume_24h' => rand(50000000, 200000000),
            'avg_profit' => rand(500, 2000),
            'top_exchanges' => [
                'Uniswap vs SushiSwap' => rand(15, 30),
                'Curve vs Uniswap' => rand(10, 20),
                '1inch vs Direct' => rand(5, 15)
            ]
        ];
    }
    
    // Utility methods
    private function getGasHistory($hours) {
        // Return mock gas price history
        $history = [];
        $currentTime = time();
        
        for ($i = 0; $i < $hours * 4; $i++) { // 4 data points per hour
            $history[] = [
                'timestamp' => $currentTime - ($i * 900), // Every 15 minutes
                'gas_price' => 15 + rand(0, 10)
            ];
        }
        
        return array_reverse($history);
    }
    
    private function cacheGasData($gasData) {
        if (!$this->db) return;
        
        try {
            $stmt = $this->db->prepare("
                INSERT INTO eth_gas_history (timestamp, gas_price, recommended) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute([
                $gasData['timestamp'],
                json_encode($gasData['gas_price']),
                json_encode($gasData['recommended'])
            ]);
        } catch (PDOException $e) {
            error_log("Failed to cache gas data: " . $e->getMessage());
        }
    }
    
    private function getAlertsFromDatabase() {
        // Get alerts from database
        if (!$this->db) return $this->generateMockAlerts();
        
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM eth_security_alerts 
                ORDER BY created_at DESC 
                LIMIT 20
            ");
            $stmt->execute();
            
            $alerts = [];
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $alerts[] = [
                    'id' => $row['id'],
                    'type' => $row['type'],
                    'severity' => $row['severity'],
                    'message' => $row['message'],
                    'created_at' => $row['created_at']
                ];
            }
            
            return $alerts;
        } catch (PDOException $e) {
            return $this->generateMockAlerts();
        }
    }
    
    private function generateMockAlerts() {
        return [
            [
                'id' => 1,
                'type' => 'MEV_BOT',
                'severity' => 'warning',
                'message' => 'New MEV bot detected targeting Uniswap v3',
                'created_at' => time() - 120
            ],
            [
                'id' => 2,
                'type' => 'GAS_SPIKE',
                'severity' => 'warning',
                'message' => 'Gas price increased 40% in the last 10 minutes',
                'created_at' => time() - 300
            ],
            [
                'id' => 3,
                'type' => 'STAKING_UPDATE',
                'severity' => 'info',
                'message' => 'Staking contract updated to version 4.2',
                'created_at' => time() - 3600
            ]
        ];
    }
    
    // Helper methods for gas levels
    private function getSafeGasPrice() { return 20 + rand(0, 10); }
    private function getStandardGasPrice() { return 15 + rand(0, 10); }
    private function getFastGasPrice() { return 10 + rand(0, 10); }
    
    // Helper methods for staking data
    private function getTotalStaked() { return 31800000 + rand(-500000, 500000); }
    private function getCurrentAPR() { return 4.0 + (rand(0, 50) / 10); }
    private function getActiveValidatorCount() { return 900000 + rand(-10000, 10000); }
    private function getPendingValidatorCount() { return 5000 + rand(-1000, 1000); }
    
    // Helper methods for L2 networks
    private function getArbitrumGasPrice() { return (rand(1, 100) / 1000); }
    private function getArbitrumTPS() { return 4000 + rand(-1000, 1000); }
    private function getArbitrumVolume() { return rand(400000000, 600000000); }
    
    private function getOptimismGasPrice() { return (rand(1, 200) / 1000); }
    private function getOptimismTPS() { return 2500 + rand(-500, 500); }
    private function getOptimismVolume() { return rand(150000000, 250000000); }
    
    private function getBaseGasPrice() { return (rand(1, 50) / 1000); }
    private function getBaseTPS() { return 12000 + rand(-2000, 2000); }
    private function getBaseVolume() { return rand(100000000, 200000000); }
    
    // Helper methods for network data
    private function getCurrentBlockNumber() { return 18500000 + rand(0, 1000); }
    private function getAverageBlockTime() { return 12 + (rand(-30, 30) / 10); }
    private function getGasUsed24h() { return rand(140000000000, 160000000000); }
    
    // Response helper methods
    private function successResponse($data) {
        return json_encode([
            'success' => true,
            'data' => $data,
            'timestamp' => time()
        ]);
    }
    
    private function errorResponse($message, $code = 400) {
        http_response_code($code);
        return json_encode([
            'success' => false,
            'error' => $message,
            'timestamp' => time()
        ]);
    }
}

// Initialize and handle the API request
$api = new EthereumIntelligenceAPI();
$response = $api->handleRequest();

echo $response;