#!/bin/bash

# =====================================================
# MetaChain Database Setup Script
# Database: nvdaxcom_galliumhash_metachain
# Purpose: Initialize multi-chain intelligence database
# =====================================================

echo "🚀 Setting up MetaChain Database..."
echo "=================================="

# Database configuration
DB_NAME="nvdaxcom_galliumhash_metachain"
DB_USER="nvdaxcom_galliumhash_metabuilder_user"
DB_PASS="32TbgulnE7YOd2G0e5"
DB_HOST="localhost"

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

print_error() {
    echo -e "${RED}❌ $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

# Check if MySQL is installed
print_info "Checking MySQL installation..."
if ! command -v mysql &> /dev/null; then
    print_error "MySQL is not installed. Please install MySQL first."
    exit 1
fi
print_status "MySQL found"

# Check if database exists
print_info "Checking if database exists..."
DB_EXISTS=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" -e "SHOW DATABASES LIKE '$DB_NAME';" 2>/dev/null | grep "$DB_NAME" || echo "")

if [ -z "$DB_EXISTS" ]; then
    print_info "Creating database '$DB_NAME'..."
    mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" -e "CREATE DATABASE $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
    
    if [ $? -eq 0 ]; then
        print_status "Database created successfully"
    else
        print_error "Failed to create database"
        exit 1
    fi
else
    print_warning "Database '$DB_NAME' already exists"
    print_info "Do you want to continue? (This will overwrite existing data) [y/N]"
    read -r response
    if [[ ! "$response" =~ ^[Yy]$ ]]; then
        print_info "Setup cancelled"
        exit 0
    fi
fi

# Execute the database setup
print_info "Executing database schema setup..."
mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" < /workspace/space.galliumhash.com/metachain/backend/database.sql

if [ $? -eq 0 ]; then
    print_status "Database schema created successfully"
else
    print_error "Failed to execute database schema"
    exit 1
fi

# Verify setup
print_info "Verifying database setup..."

# Check tables
TABLES=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SHOW TABLES;" 2>/dev/null | tail -n +2 | wc -l)
print_status "Created $TABLES tables"

# Check initial data
CHAINS=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SELECT COUNT(*) FROM chains;" 2>/dev/null | tail -n +2)
MINING_METHODS=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SELECT COUNT(*) FROM mining_methods;" 2>/dev/null | tail -n +2)
ACHIEVEMENTS=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SELECT COUNT(*) FROM achievements;" 2>/dev/null | tail -n +2)
EARNING_RULES=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SELECT COUNT(*) FROM earning_rules;" 2>/dev/null | tail -n +2)

print_status "Inserted $CHAINS chains"
print_status "Inserted $MINING_METHODS mining methods"
print_status "Inserted $ACHIEVEMENTS achievements"
print_status "Inserted $EARNING_RULES earning rules"

# Create database backup setup
print_info "Setting up database backup configuration..."
BACKUP_DIR="/workspace/space.galliumhash.com/metachain/backend/backups"
mkdir -p "$BACKUP_DIR"

# Create backup script
cat > "$BACKUP_DIR/backup.sh" << 'EOF'
#!/bin/bash

# MetaChain Database Backup Script
DB_NAME="nvdaxcom_galliumhash_metachain"
DB_USER="nvdaxcom_galliumhash_metabuilder_user"
DB_PASS="32TbgulnE7YOd2G0e5"
DB_HOST="localhost"
BACKUP_DIR="/workspace/space.galliumhash.com/metachain/backend/backups"

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_FILE="$BACKUP_DIR/metachain_backup_$DATE.sql"

# Create backup
mysqldump -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" > "$BACKUP_FILE"

# Compress backup
gzip "$BACKUP_FILE"

# Keep only last 30 days of backups
find "$BACKUP_DIR" -name "metachain_backup_*.sql.gz" -mtime +30 -delete

echo "Backup completed: ${BACKUP_FILE}.gz"
EOF

chmod +x "$BACKUP_DIR/backup.sh"
print_status "Backup script created: $BACKUP_DIR/backup.sh"

# Create database cleanup script for development
cat > "/workspace/space.galliumhash.com/metachain/backend/cleanup_dev.sql" << 'EOF'
-- =====================================================
-- MetaChain Database Cleanup (Development Only)
-- Use this script to reset the database for testing
-- =====================================================

USE nvdaxcom_galliumhash_metachain;

-- Disable foreign key checks
SET FOREIGN_KEY_CHECKS = 0;

-- Truncate all data tables (keep structure)
TRUNCATE TABLE user_mining_activities;
TRUNCATE TABLE user_predictions;
TRUNCATE TABLE token_transactions;
TRUNCATE TABLE user_achievements;
TRUNCATE TABLE global_leaderboard;
TRUNCATE TABLE chain_leaderboards;
TRUNCATE TABLE method_leaderboards;
TRUNCATE TABLE ethereum_intelligence;
TRUNCATE TABLE solana_intelligence;
TRUNCATE TABLE cross_chain_analytics;
TRUNCATE TABLE defi_protocols;
TRUNCATE TABLE nft_markets;
TRUNCATE TABLE api_cache;

-- Clear user data (keep default users for testing)
DELETE FROM users WHERE username NOT IN ('admin', 'testuser');

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1;

-- Recreate test users
INSERT INTO users (username, email, wallet_address, total_tokens, tier) VALUES
('admin', 'admin@galliumhash.com', '0x0000000000000000000000000000000000000000', 1000.00000000, 'builder'),
('testuser', 'test@galliumhash.com', '0x1234567890123456789012345678901234567890', 50.00000000, 'engaged');

-- Update leaderboards
TRUNCATE TABLE global_leaderboard;
INSERT INTO global_leaderboard (user_id, total_tokens, mining_streak_days, prediction_accuracy, global_rank)
SELECT id, total_tokens, 0, 0.00, ROW_NUMBER() OVER (ORDER BY total_tokens DESC)
FROM users;

print "Database cleaned for development - test data restored";
EOF

print_status "Development cleanup script created"

# Final setup verification
print_info "Running final setup verification..."

# Check database size
DB_SIZE=$(mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" -e "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) AS 'DB Size in MB' FROM information_schema.tables WHERE table_schema='$DB_NAME';" 2>/dev/null | tail -n +2)
print_status "Database size: $DB_SIZE MB"

# Check for any errors in the log
print_info "Setup completed successfully! 🎉"
echo ""
echo "📊 Database Summary:"
echo "==================="
echo "🗄️  Database: $DB_NAME"
echo "📋 Tables: $TABLES"
echo "⛓️  Chains: $CHAINS"
echo "⛏️  Mining Methods: $MINING_METHODS"
echo "🏆 Achievements: $ACHIEVEMENTS"
echo "💰 Earning Rules: $EARNING_RULES"
echo "💾 Database Size: $DB_SIZE MB"
echo ""
echo "🔧 Available Scripts:"
echo "===================="
echo "💾 Backup: $BACKUP_DIR/backup.sh"
echo "🧹 Cleanup: /workspace/space.galliumhash.com/metachain/backend/cleanup_dev.sql"
echo ""
echo "🚀 Next Steps:"
echo "============="
echo "1. Configure API keys for each chain (Ethereum, Solana, etc.)"
echo "2. Set up cron jobs for data collection"
echo "3. Test each intelligence hub functionality"
echo "4. Deploy metachain frontend components"
echo "5. Start processing user activities and predictions"
echo ""
print_status "MetaChain database setup complete!"

# Create a setup verification query
mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" "$DB_NAME" << 'EOF'
-- Verify core tables are working
SELECT 'Users Table' as verification, COUNT(*) as count FROM users
UNION ALL
SELECT 'Chains Table', COUNT(*) FROM chains
UNION ALL
SELECT 'Mining Methods', COUNT(*) FROM mining_methods
UNION ALL
SELECT 'Prediction Markets', COUNT(*) FROM prediction_markets
UNION ALL
SELECT 'Token Transactions', COUNT(*) FROM token_transactions
UNION ALL
SELECT 'Global Leaderboard', COUNT(*) FROM global_leaderboard
UNION ALL
SELECT 'Achievements', COUNT(*) FROM achievements;
EOF