# MetaChain Database System Documentation

## Overview

The MetaChain database system (`nvdaxcom_galliumhash_metachain`) is the foundation for the Multi-Chain Intelligence Center. It supports universal mining, prediction markets, token economy, and cross-chain analytics across all supported blockchain networks.

## Database Architecture

### Core Components

1. **User Management System**
   - Multi-chain user profiles with wallet integration
   - Verification levels (basic, verified, premium)
   - Tier progression (casual, engaged, power, builder)
   - Token balance tracking

2. **Universal Mining Framework**
   - 8 universal mining methods for non-hardware mining
   - Activity-based token earning system
   - Method-specific leaderboards and achievements

3. **Prediction Market System**
   - Chain-specific prediction markets
   - User predictions with confidence scoring
   - Automated result resolution and payouts

4. **Token Economy Engine**
   - Complete transaction history
   - Dynamic earning rules and bonuses
   - Automated tier progression

5. **Intelligence Hub Data**
   - Real-time blockchain analytics
   - Cross-chain arbitrage detection
   - DeFi and NFT market tracking

6. **Leaderboard & Achievement System**
   - Global rankings
   - Chain-specific rankings
   - Method-specific rankings
   - Badge-based achievements

## Database Schema

### Tables Overview

#### User Management
- **`users`** - Core user profiles and metadata
- **`user_verifications`** - KYC and verification tracking

#### Chain Management
- **`chains`** - Supported blockchain networks
- **Chain-specific intelligence tables:**
  - `ethereum_intelligence` - ETH gas, MEV, staking data
  - `solana_intelligence` - SOL TPS, congestion, validator data
  - `cross_chain_analytics` - Arbitrage opportunities
  - `defi_protocols` - DeFi protocol analytics
  - `nft_markets` - NFT marketplace data

#### Universal Mining
- **`mining_methods`** - 8 universal mining methods
- **`user_mining_activities`** - Activity tracking
- **`mining_achievements`** - Achievement definitions
- **`user_achievements`** - User progress tracking

#### Prediction Markets
- **`prediction_markets`** - Market definitions
- **`user_predictions`** - User participation
- **`prediction_results`** - Result resolution

#### Token Economy
- **`token_transactions`** - Complete transaction history
- **`earning_rules`** - Dynamic bonus system
- **`user_tiers`** - Tier progression tracking

#### Leaderboards
- **`global_leaderboard`** - Overall rankings
- **`chain_leaderboards`** - Chain-specific rankings
- **`method_leaderboards`** - Method-specific rankings

#### Performance & Caching
- **`api_cache`** - API response caching
- **`system_settings`** - Configuration management

## Universal Mining Methods

### 1. Prediction Mining (`prediction_mining`)
- **Description**: Earn tokens by making accurate market predictions
- **Rate**: 5.0 tokens per prediction
- **Requirements**: Chain-specific prediction markets
- **Leaderboard**: Highest prediction accuracy

### 2. Price Tracking Mining (`price_tracking_mining`)
- **Description**: Engage with price alerts and tracking activities
- **Rate**: 2.5 tokens per engagement
- **Requirements**: Active price alert subscriptions
- **Leaderboard**: Most price tracking engagements

### 3. Opportunity Scouting Mining (`opportunity_scouting_mining`)
- **Description**: Hunt for arbitrage and alpha opportunities
- **Rate**: 10.0 tokens per opportunity identified
- **Requirements**: Cross-chain analytics capability
- **Leaderboard**: Most profitable opportunities found

### 4. Social Intelligence Mining (`social_intelligence_mining`)
- **Description**: Provide community intelligence and insights
- **Rate**: 7.5 tokens per contribution
- **Requirements**: Verified community status
- **Leaderboard**: Most upvoted social contributions

### 5. Cross-Chain Arbitrage Mining (`cross_chain_arbitrage_mining`)
- **Description**: Identify and report cross-chain price differences
- **Rate**: 15.0 tokens per verified opportunity
- **Requirements**: Multi-chain wallet access
- **Leaderboard**: Highest arbitrage volume

### 6. Educational Mining (`educational_mining`)
- **Description**: Create educational content and tutorials
- **Rate**: 3.0 tokens per content piece
- **Requirements**: Educational content creation
- **Leaderboard**: Most educational content created

### 7. Data Labeling Mining (`data_labeling_mining`)
- **Description**: Contribute to AI training datasets
- **Rate**: 4.0 tokens per labeled data point
- **Requirements**: Data labeling qualification
- **Leaderboard**: Most accurate data labels

### 8. Community Moderation Mining (`community_moderation_mining`)
- **Description**: Moderate content and maintain community standards
- **Rate**: 6.0 tokens per moderation action
- **Requirements**: Moderator privileges
- **Leaderboard**: Highest moderation quality score

## Supported Chains

| Chain | Symbol | Explorer | Special Features |
|-------|--------|----------|------------------|
| Bitcoin | BTC | blockstream.info | Original mining data |
| Ethereum | ETH | etherscan.io | Gas, MEV, Staking, L2s |
| Solana | SOL | explorer.solana.com | TPS, Congestion, DeFi |
| Polygon | MATIC | polygonscan.com | Layer 2 scaling |
| Arbitrum | ARB | arbiscan.io | Layer 2 DeFi |
| Optimism | OP | optimistic.etherscan.io | Layer 2 gaming |
| Base | BASE | basescan.org | Coinbase Layer 2 |
| BNB Chain | BNB | bscscan.com | Binance Smart Chain |

## Token Economy

### Tier System

1. **Casual (0-50 tokens)**
   - Basic mining activities
   - Access to simple prediction markets
   - Community features

2. **Engaged (50-200 tokens)**
   - Advanced mining methods
   - Higher prediction limits
   - Early access to new features

3. **Power (200-1000 tokens)**
   - Premium mining methods
   - Unlimited predictions
   - Access to exclusive markets

4. **Builder (1000+ tokens)**
   - All mining methods
   - Developer API access
   - Governance participation

### Earning Rules

- **Daily Login**: 5 tokens
- **Mining Streak (7 days)**: 25 tokens bonus
- **Prediction Accuracy (>80%)**: 15 tokens bonus
- **Cross-Chain Pioneer**: 50 tokens one-time
- **Early Adopter (Top 1000)**: 100 tokens one-time

## Installation & Setup

### Prerequisites
- MySQL 8.0 or higher
- Access to database server
- 1GB+ available disk space

### Quick Setup

```bash
# Make setup script executable (if possible)
chmod +x /workspace/space.galliumhash.com/metachain/backend/setup.sh

# Run database setup
./backend/setup.sh
```

### Manual Setup

```bash
# Create database
mysql -u username -p -e "CREATE DATABASE nvdaxcom_galliumhash_metachain CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

# Import schema
mysql -u username -p nvdaxcom_galliumhash_metachain < backend/database.sql
```

### Verification

```sql
-- Check database setup
SELECT 
    (SELECT COUNT(*) FROM users) as users,
    (SELECT COUNT(*) FROM chains) as chains,
    (SELECT COUNT(*) FROM mining_methods) as mining_methods,
    (SELECT COUNT(*) FROM achievements) as achievements;
```

## API Integration

### Database Connection

```php
<?php
$host = 'localhost';
$dbname = 'nvdaxcom_galliumhash_metachain';
$username = 'nvdaxcom_galliumhash_metabuilder_user';
$password = '32TbgulnE7YOd2G0e5';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}
?>
```

### Sample Queries

#### Get User Activity Summary
```sql
SELECT 
    username,
    tier,
    total_tokens,
    total_activities,
    total_predictions,
    avg_prediction_confidence,
    total_mining_earnings
FROM user_activity_summary
ORDER BY total_tokens DESC
LIMIT 10;
```

#### Get Chain Analytics
```sql
SELECT 
    chain_name,
    active_users,
    total_mining_tokens,
    total_predictions,
    avg_prediction_accuracy
FROM chain_analytics_summary
ORDER BY active_users DESC;
```

#### Get Cross-Chain Arbitrage Opportunities
```sql
SELECT 
    c1.name as chain1,
    c2.name as chain2,
    asset_symbol,
    price_chain1,
    price_chain2,
    price_difference_percentage,
    timestamp
FROM cross_chain_analytics ca
JOIN chains c1 ON ca.chain1_id = c1.id
JOIN chains c2 ON ca.chain2_id = c2.id
WHERE arbitrage_opportunity = TRUE
    AND price_difference_percentage > 5.0
ORDER BY price_difference_percentage DESC
LIMIT 10;
```

## Data Collection Setup

### Ethereum Intelligence
```bash
# Add to crontab for every 30 seconds
*/1 * * * * curl -s "https://api.etherscan.io/api?module=gastracker&action=gasoracle&apikey=YOUR_API_KEY" | \
jq -r '.result' > /tmp/eth_gas.json

# Process with your PHP script
*/1 * * * * php /path/to/ethereum-intelligence.php
```

### Cross-Chain Analytics
```bash
# Every 5 minutes
*/5 * * * * php /path/to/cross-chain-analytics.php
```

### Leaderboard Updates
```bash
# Every 30 minutes
*/30 * * * * php /path/to/update-leaderboards.php
```

## Performance Optimization

### Indexes
- All foreign key columns are indexed
- Timestamp columns indexed for time-series queries
- User activity queries optimized with composite indexes

### Caching
- API responses cached for 5 minutes
- Leaderboard updates cached for 30 minutes
- User activity summaries cached for 1 hour

### Views
- `user_activity_summary` - Pre-calculated user metrics
- `chain_analytics_summary` - Chain performance metrics

## Backup & Maintenance

### Backup Script
```bash
# Manual backup
./backend/backups/backup.sh

# Automated backup (add to crontab)
0 2 * * * /workspace/space.galliumhash.com/metachain/backend/backups/backup.sh
```

### Development Cleanup
```bash
# Reset database for testing
mysql -u username -p nvdaxcom_galliumhash_metachain < backend/cleanup_dev.sql
```

### Monitoring Queries
```sql
-- Check database size
SELECT 
    table_schema as 'Database',
    ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) as 'Size (MB)'
FROM information_schema.tables 
WHERE table_schema = 'nvdaxcom_galliumhash_metachain'
GROUP BY table_schema;

-- Check table sizes
SELECT 
    table_name as 'Table',
    ROUND(((data_length + index_length) / 1024 / 1024), 1) as 'Size (MB)'
FROM information_schema.TABLES 
WHERE table_schema = 'nvdaxcom_galliumhash_metachain'
ORDER BY (data_length + index_length) DESC;
```

## Security Considerations

### Access Control
- Database user has limited privileges
- Write access only to necessary tables
- Read-only access to reference tables

### Data Validation
- All user inputs sanitized
- SQL injection prevention with prepared statements
- Rate limiting on activity submissions

### API Security
- API keys stored securely
- Request rate limiting
- Data validation on all inputs

## Troubleshooting

### Common Issues

1. **Connection Failed**
   - Check database server is running
   - Verify credentials
   - Check firewall settings

2. **Table Creation Failed**
   - Ensure MySQL version compatibility
   - Check disk space
   - Verify user privileges

3. **Performance Issues**
   - Monitor slow queries
   - Check index usage
   - Consider query optimization

4. **Data Collection Failures**
   - Verify API keys
   - Check network connectivity
   - Monitor API rate limits

### Log Files
- Database errors: `/var/log/mysql/error.log`
- Application logs: `/path/to/metachain/logs/`
- API logs: `/path/to/metachain/logs/api/`

## Support & Maintenance

### Regular Tasks
- Monitor database performance
- Update API keys as needed
- Clean expired cache entries
- Backup verification

### Monthly Tasks
- Analyze query performance
- Update statistics
- Review and optimize indexes
- Security audit

### Updates
- Schema changes tracked in version control
- Migration scripts for updates
- Backward compatibility maintained

## Version History

### v1.0 (2025-10-24)
- Initial database schema
- Complete user management system
- Universal mining framework
- Prediction market system
- Token economy engine
- Intelligence hub data structures
- Leaderboard and achievement systems

---

**Database System Ready for Production Deployment** 🚀

For technical support or questions about the database schema, refer to the inline comments in the SQL files or contact the development team.