#!/usr/bin/env php
<?php
/**
 * MetaBuilder Automated Data Collection
 * 
 * This script runs MetaBuilder collectors on a schedule
 * cPanel Optimized: Uses existing MetaBuilder infrastructure
 * 
 * CRON SETUP:
 * Add to cPanel cron jobs:
 * */10 * * * * /usr/bin/php /path/to/core.galliumhash.com/automation/metabuilder-collector.php
 */

date_default_timezone_set('UTC');
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Paths
$metabuilderPath = __DIR__ . '/../metabuilder';
$logFile = __DIR__ . '/logs/metabuilder-collector.log';

// Create logs directory
if (!is_dir(dirname($logFile))) {
    mkdir(dirname($logFile), 0755, true);
}

/**
 * Log message with timestamp
 */
function logMessage($message, $level = 'INFO') {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message\n";
    echo "[$timestamp] $message\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Check if collection should run based on last run time
 */
function shouldRunCollection($intervalMinutes = 10) {
    $lockFile = __DIR__ . '/logs/metabuilder-last-run.lock';
    $lastRun = 0;
    
    if (file_exists($lockFile)) {
        $lastRun = (int) file_get_contents($lockFile);
    }
    
    $now = time();
    $timeSinceLastRun = $now - $lastRun;
    $intervalSeconds = $intervalMinutes * 60;
    
    return $timeSinceLastRun >= $intervalSeconds;
}

/**
 * Update last run timestamp
 */
function updateLastRun() {
    $lockFile = __DIR__ . '/logs/metabuilder-last-run.lock';
    file_put_contents($lockFile, time());
}

/**
 * Run MetaBuilder collection
 */
function runMetaBuilderCollection() {
    global $metabuilderPath;
    
    logMessage("Starting MetaBuilder data collection...");
    
    try {
        // Change to MetaBuilder directory
        chdir($metabuilderPath);
        
        // Run collection
        $output = [];
        $returnCode = 0;
        exec("php metabuilder.php --action=collect 2>&1", $output, $returnCode);
        
        $outputStr = implode("\n", $output);
        
        if ($returnCode === 0) {
            logMessage("✅ MetaBuilder collection completed successfully");
            logMessage("Output: " . substr($outputStr, 0, 500) . "...");
            return [
                'success' => true,
                'output' => $outputStr,
                'timestamp' => date('c')
            ];
        } else {
            logMessage("❌ MetaBuilder collection failed with code $returnCode", 'ERROR');
            logMessage("Error: " . substr($outputStr, 0, 500), 'ERROR');
            return [
                'success' => false,
                'error' => $outputStr,
                'return_code' => $returnCode,
                'timestamp' => date('c')
            ];
        }
        
    } catch (Exception $e) {
        logMessage("❌ Exception in MetaBuilder collection: " . $e->getMessage(), 'ERROR');
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

/**
 * Get system health status
 */
function getHealthStatus() {
    try {
        chdir(__DIR__ . '/../metabuilder');
        $output = [];
        exec("php metabuilder.php --action=health 2>&1", $output, $returnCode);
        
        $healthData = implode("\n", $output);
        
        // Try to parse JSON response
        $health = json_decode($healthData, true);
        if ($health) {
            return $health;
        }
        
        return [
            'status' => 'unknown',
            'raw_output' => $healthData,
            'timestamp' => date('c')
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

// Main execution
try {
    $action = $argv[1] ?? 'run';
    
    switch ($action) {
        case 'run':
            logMessage("MetaBuilder Collector starting...");
            
            // Check if we should run based on interval
            if (!shouldRunCollection(10)) {
                $timeSinceLast = time() - (int) file_get_contents(__DIR__ . '/logs/metabuilder-last-run.lock');
                $remaining = (10 * 60) - $timeSinceLast;
                logMessage("Skipping collection - ran recently. " . gmdate("H:i:s", $remaining) . " remaining until next run.");
                exit(0);
            }
            
            // Run collection
            $result = runMetaBuilderCollection();
            
            // Update last run timestamp
            updateLastRun();
            
            // Log results
            logMessage("Collection result: " . ($result['success'] ? 'SUCCESS' : 'FAILED'));
            
            // Output JSON for external monitoring
            echo json_encode($result, JSON_PRETTY_PRINT) . "\n";
            
            // Return appropriate exit code
            exit($result['success'] ? 0 : 1);
            
        case 'health':
            $health = getHealthStatus();
            echo json_encode($health, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'test':
            logMessage("MetaBuilder Collector - Test mode");
            
            // Test MetaBuilder availability
            $health = getHealthStatus();
            echo "MetaBuilder Health Check:\n";
            echo "- Status: " . ($health['status'] ?? 'unknown') . "\n";
            
            if (isset($health['checks'])) {
                foreach ($health['checks'] as $check => $status) {
                    echo "- $check: $status\n";
                }
            }
            break;
            
        default:
            echo "Usage: php metabuilder-collector.php [run|health|test]\n";
            echo "  run    - Run data collection\n";
            echo "  health - Check system health\n";
            echo "  test   - Test system availability\n";
    }
    
} catch (Exception $e) {
    logMessage("Fatal error: " . $e->getMessage(), 'ERROR');
    echo json_encode([
        'error' => true,
        'message' => $e->getMessage(),
        'timestamp' => date('c')
    ], JSON_PRETTY_PRINT) . "\n";
    exit(1);
}