#!/usr/bin/env php
<?php
/**
 * Ethereum GasCollector Automated Data Collection
 * 
 * This script runs Ethereum GasCollector on a schedule
 * cPanel Optimized: Uses existing GasCollector infrastructure
 * 
 * CRON SETUP:
 * Add to cPanel cron jobs:
 * */5 * * * * /usr/bin/php /path/to/core.galliumhash.com/automation/ethereum-gas-collector.php
 */

date_default_timezone_set('UTC');
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Paths
$gasCollectorPath = __DIR__ . '/../metachain/ethereum/builder/collectors';
$logFile = __DIR__ . '/logs/ethereum-gas-collector.log';

// Create logs directory
if (!is_dir(dirname($logFile))) {
    mkdir(dirname($logFile), 0755, true);
}

/**
 * Log message with timestamp
 */
function logMessage($message, $level = 'INFO') {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message\n";
    echo "[$timestamp] $message\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Check if collection should run based on interval
 */
function shouldRunCollection($intervalMinutes = 5) {
    $lockFile = __DIR__ . '/logs/ethereum-gas-last-run.lock';
    $lastRun = 0;
    
    if (file_exists($lockFile)) {
        $lastRun = (int) file_get_contents($lockFile);
    }
    
    $now = time();
    $timeSinceLastRun = $now - $lastRun;
    $intervalSeconds = $intervalMinutes * 60;
    
    return $timeSinceLastRun >= $intervalSeconds;
}

/**
 * Update last run timestamp
 */
function updateLastRun() {
    $lockFile = __DIR__ . '/logs/ethereum-gas-last-run.lock';
    file_put_contents($lockFile, time());
}

/**
 * Run Ethereum GasCollector
 */
function runGasCollector() {
    global $gasCollectorPath;
    
    logMessage("Starting Ethereum GasCollector...");
    
    try {
        // Change to GasCollector directory
        chdir($gasCollectorPath);
        
        // Run GasCollector
        $output = [];
        $returnCode = 0;
        exec("php GasCollector.php 2>&1", $output, $returnCode);
        
        $outputStr = implode("\n", $output);
        
        if ($returnCode === 0 || strpos($outputStr, 'success') !== false) {
            logMessage("✅ Ethereum GasCollector completed successfully");
            
            // Extract key information from output
            $lines = explode("\n", $outputStr);
            foreach ($lines as $line) {
                if (strpos($line, 'gas') !== false && strpos($line, 'Gwei') !== false) {
                    logMessage("Latest gas data: " . trim($line));
                    break;
                }
            }
            
            return [
                'success' => true,
                'output' => $outputStr,
                'timestamp' => date('c')
            ];
        } else {
            logMessage("❌ Ethereum GasCollector failed with code $returnCode", 'ERROR');
            logMessage("Error: " . substr($outputStr, 0, 500), 'ERROR');
            return [
                'success' => false,
                'error' => $outputStr,
                'return_code' => $returnCode,
                'timestamp' => date('c')
            ];
        }
        
    } catch (Exception $e) {
        logMessage("❌ Exception in GasCollector: " . $e->getMessage(), 'ERROR');
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

/**
 * Test GasCollector functionality
 */
function testGasCollector() {
    global $gasCollectorPath;
    
    try {
        chdir($gasCollectorPath);
        
        // Test direct instantiation
        $testCode = '
        try {
            require_once "GasCollector.php";
            $collector = new GasCollector();
            echo "GasCollector instantiated successfully\n";
            
            // Test database connection
            if ($collector->getLastCollectionTime()) {
                echo "Database connection working\n";
            }
            
            echo "All tests passed\n";
            exit(0);
        } catch (Exception $e) {
            echo "Test failed: " . $e->getMessage() . "\n";
            exit(1);
        }
        ';
        
        file_put_contents('/tmp/gas_collector_test.php', $testCode);
        
        $output = [];
        $returnCode = 0;
        exec("php /tmp/gas_collector_test.php 2>&1", $output, $returnCode);
        
        unlink('/tmp/gas_collector_test.php');
        
        $testOutput = implode("\n", $output);
        
        return [
            'success' => $returnCode === 0,
            'output' => $testOutput,
            'return_code' => $returnCode
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Check GasCollector health via database
 */
function checkGasCollectorHealth() {
    try {
        // Load environment variables
        if (file_exists(__DIR__ . '/../../.env')) {
            $lines = file(__DIR__ . '/../../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach ($lines as $line) {
                if (strpos($line, '=') !== false && !empty(trim($line)) && !str_starts_with($line, '#')) {
                    list($key, $value) = explode('=', $line, 2);
                    $_ENV[trim($key)] = trim($value);
                }
            }
        }
        
        // Connect to database
        $dsn = "mysql:host=" . ($_ENV['DB_HOST'] ?? 'localhost') . 
               ";dbname=" . ($_ENV['DB_NAME'] ?? 'nvdaxcom_galliumhash_metachain') . 
               ";charset=utf8mb4";
        $username = $_ENV['DB_USERNAME'] ?? 'nvdaxcom_galliumhash_metabuilder_user';
        $password = $_ENV['DB_PASSWORD'] ?? '32TbgulnE7YOd2G0e5';
        
        $db = new PDO($dsn, $username, $password);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Check last gas collection
        $stmt = $db->query("SELECT MAX(timestamp) as last_collection, COUNT(*) as total_records 
                           FROM ethereum_intelligence");
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $lastCollection = strtotime($stats['last_collection']);
        $timeSinceLast = time() - $lastCollection;
        $isFresh = $timeSinceLast < 600; // Fresh if less than 10 minutes
        
        return [
            'status' => $isFresh ? 'fresh' : 'stale',
            'last_collection_minutes' => round($timeSinceLast / 60, 1),
            'total_records' => $stats['total_records'],
            'database' => 'connected',
            'timestamp' => date('c')
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'error' => $e->getMessage(),
            'database' => 'disconnected',
            'timestamp' => date('c')
        ];
    }
}

// Main execution
try {
    $action = $argv[1] ?? 'run';
    
    switch ($action) {
        case 'run':
            logMessage("Ethereum GasCollector starting...");
            
            // Check if we should run based on interval
            if (!shouldRunCollection(5)) {
                $timeSinceLast = time() - (int) file_get_contents(__DIR__ . '/logs/ethereum-gas-last-run.lock');
                $remaining = (5 * 60) - $timeSinceLast;
                logMessage("Skipping collection - ran recently. " . gmdate("H:i:s", $remaining) . " remaining until next run.");
                exit(0);
            }
            
            // Run GasCollector
            $result = runGasCollector();
            
            // Update last run timestamp
            updateLastRun();
            
            // Log results
            logMessage("Collection result: " . ($result['success'] ? 'SUCCESS' : 'FAILED'));
            
            // Output JSON for external monitoring
            echo json_encode($result, JSON_PRETTY_PRINT) . "\n";
            
            // Return appropriate exit code
            exit($result['success'] ? 0 : 1);
            
        case 'health':
            $health = checkGasCollectorHealth();
            echo json_encode($health, JSON_PRETTY_PRINT) . "\n";
            break;
            
        case 'test':
            logMessage("Ethereum GasCollector - Test mode");
            
            // Test GasCollector functionality
            $test = testGasCollector();
            echo "GasCollector Test:\n";
            echo "- " . ($test['success'] ? 'PASS' : 'FAIL') . "\n";
            
            if ($test['output']) {
                echo "- Output: " . trim($test['output']) . "\n";
            }
            
            // Check database health
            $health = checkGasCollectorHealth();
            echo "- Database: " . $health['database'] . "\n";
            echo "- Data status: " . $health['status'] . "\n";
            echo "- Last collection: " . $health['last_collection_minutes'] . " minutes ago\n";
            break;
            
        default:
            echo "Usage: php ethereum-gas-collector.php [run|health|test]\n";
            echo "  run   - Run gas data collection\n";
            echo "  health - Check system health\n";
            echo "  test   - Test system functionality\n";
    }
    
} catch (Exception $e) {
    logMessage("Fatal error: " . $e->getMessage(), 'ERROR');
    echo json_encode([
        'error' => true,
        'message' => $e->getMessage(),
        'timestamp' => date('c')
    ], JSON_PRETTY_PRINT) . "\n";
    exit(1);
}